<?php

/**
 * WC_Gateway_ChargeAnywhere class
 *
 * @author   Rishabh Rajvanshi <rrajvan@chargeanywhere.com>
 * @package  WooCommerce Charge Anywhere Payments Gateway
 * @since    1.0.0
 */

// Exit if accessed directly.
if (! defined('ABSPATH')) {
   exit;
}

/**
 * Charge Anywhere Gateway.
 *
 * @class    WC_Gateway_ChargeAnywhere
 * @version  4.2.0
 */
class WC_Gateway_ChargeAnywhere extends WC_Payment_Gateway
{
   protected $msg = array();
   private static $initialized = true;
   private $current_refund_order_id = null;
   public function __construct()
   {
      $this->id               = 'chargeanywhere';
      $this->method_title     = __('Charge Anywhere', 'chargeanywhere-woocommerce');
      $this->method_description = __('Charge Anywhere Payment Gateway for WooCommerce. Accepts Credit Cards and ACH payments.', 'woocommerce-gateway-dummy');
      $this->has_fields       = false;
      $this->init_form_fields();
      $this->init_settings();
      
      // Ensure all settings have proper defaults (especially for plugin upgrades)
      $this->ensure_settings_exist();
      
      $this->title            = $this->settings['title'];
      $this->description      = $this->settings['description'];
      $this->merchant_id      = $this->settings['merchant_id'];
      $this->mode             = $this->settings['working_mode'];
      $this->transaction_mode = $this->settings['transaction_mode'];
      $this->auto_complete_orders = $this->settings['auto_complete_orders'];
      $this->terminal_id      = $this->settings['terminal_id'];
      $this->secret_key       = $this->settings['secret_key'];

      $this->accept_credit    = $this->settings['accept_credit'];
      $this->apply_credit_service = $this->settings['apply_credit_service'];
      $this->credit_service_fee = $this->settings['credit_service_fee'];
      $this->credit_service_fee_value = $this->settings['credit_service_fee_value'];

      $this->accept_ach       = $this->settings['accept_ach'];
      $this->apply_ach_service = $this->settings['apply_ach_service'];
      $this->ach_service_fee  = $this->settings['ach_service_fee'];
      $this->ach_service_fee_value = $this->settings['ach_service_fee_value'];

      $this->success_message  = $this->settings['success_message'];
      $this->failed_message   = $this->settings['failed_message'];
      $this->email_customer   = 'false';
      $this->email_merchant   = 'false';
      $this->log_request      = 'false';
      $this->redirect_message = '';

      if (isset($this->settings['email_customer']) && $this->settings['email_customer'] == 'yes')
         $this->email_customer = 'true';

      if (isset($this->settings['email_merchant']) && $this->settings['email_merchant'] == 'yes')
         $this->email_merchant = 'true';

      if (isset($this->settings['log_request']) && $this->settings['log_request'] == 'yes')
         $this->log_request    = 'true';

      if (isset($this->settings['redirect_message']))
         $this->redirect_message  = $this->settings['redirect_message'];

      $this->live_url_transaction   = 'https://www.chargeanywhere.com/APIs/PaymentFormSIP.aspx';
      $this->test_url_transaction   = 'https://webtest.chargeanywhere.com/APIs/PaymentFormSIP.aspx';

      $this->live_url_process       = 'https://www.chargeanywhere.com/APIs/PaymentFormAIP.aspx';
      $this->test_url_process       = 'https://webtest.chargeanywhere.com/APIs/PaymentFormAIP.aspx';

      $this->msg['message']   = "";
      $this->msg['class']     = "";
      $this->newrelay         = home_url() . '/wc-api/WC_Gateway_ChargeAnywhere';



      if ($this->transaction_mode === 'authorize') {
         add_action('woocommerce_order_status_completed', array($this, 'process_capture'));
         // Prevent auto-completion of virtual/downloadable products in authorize mode
         add_filter('woocommerce_payment_complete_order_status', array($this, 'override_payment_complete_status'), 10, 3);
      }
      add_action('woocommerce_api_wc_gateway_chargeanywhere', array($this, 'check_authorize_response'));
      add_action('valid-authorize-request', array(&$this, 'successful_request'));


      if (version_compare(WOOCOMMERCE_VERSION, '2.0.0', '>=')) {
         add_action('woocommerce_update_options_payment_gateways_' . $this->id, array(&$this, 'process_admin_options'));
      } else {
         add_action('woocommerce_update_options_payment_gateways', array(&$this, 'process_admin_options'));
      }
      if (self::$initialized) {
         add_action('woocommerce_receipt_chargeanywhere', array(&$this, 'receipt_page'));
      }
      
      
      $this->supports = array('products', 'refunds');
      self::$initialized = false;
   }

   /**
    * Ensure all form field settings exist with proper defaults
    * This handles plugin upgrades where new settings were added
    */
   private function ensure_settings_exist()
   {
      $settings_updated = false;
      
      // Check each form field and ensure setting exists
      foreach ($this->form_fields as $key => $field) {
         if (!isset($this->settings[$key])) {
            // Setting doesn't exist, apply default value
            $default_value = isset($field['default']) ? $field['default'] : '';
            $this->settings[$key] = $default_value;
            $settings_updated = true;
            
         }
      }
      
      // Only upgrade completely empty fee labels to descriptive defaults
      // Respect admin's intentional choices like "SF", "CF", etc.
      $fee_label_upgrades = array(
         'credit_service_fee' => 'Credit Card Service Fee',
         'credit_convenience_service_fee' => 'Credit Card Convenience Fee',
         'ach_service_fee' => 'ACH Service Fee',
         'ach_convenience_service_fee' => 'ACH Convenience Fee'
      );
      
      foreach ($fee_label_upgrades as $setting_key => $descriptive_label) {
         if (isset($this->settings[$setting_key])) {
            $current_value = trim($this->settings[$setting_key]);
            // Only upgrade if completely empty - respect admin's intentional labels
            if (empty($current_value)) {
               $this->settings[$setting_key] = $descriptive_label;
               $settings_updated = true;
            }
         }
      }
      
      // If we added any settings, save them to database
      if ($settings_updated) {
         update_option($this->get_option_key(), $this->settings);
      }
   }

   /**
    * process_refund function.
    *
    * @access public
    * @param int $order_id
    * @param float $amount
    * @param string $reason
    * @return bool|WP_Error
    */
   public function process_refund($order_id, $amount = NULL, $reason = '')
   {
      global $wpdb;
      $options = get_option('woocommerce_chargeanywhere_settings');
      $order = wc_get_order($order_id);
      
      if (!$order) {
         return new WP_Error('invalid_order', __('Invalid order ID', 'woocommerce-chargeanywhere'));
      }
      
      // Validate refund request before processing
      $validation_result = $this->validate_refund_request($order, $amount, $reason);
      if (is_wp_error($validation_result)) {
         return $validation_result;
      }
      
      // Protect against WordPress/WooCommerce interference during refund processing
      $this->protect_refund_process($order_id);
      


      $tran_meta = $order->get_meta('_chargeanywhere_transaction');
      $chargeanywherefee_details = (array) json_decode($tran_meta['chargeanywhere_fee_details']);

      $line_item_totals       = isset($_POST['line_item_totals']) ? json_decode(sanitize_text_field(wp_unslash($_POST['line_item_totals'])), true) : array();
      $line_item_tax_totals   = isset($_POST['line_item_tax_totals']) ? json_decode(sanitize_text_field(wp_unslash($_POST['line_item_tax_totals'])), true) : array();

      // Enable smart item-level validation that works with pro-rata fee calculations
      $refund_validation = $this->prevent_double_refund($order, $line_item_totals, $amount);
      if (is_wp_error($refund_validation)) {
         $this->cleanup_refund_protection();
         return $refund_validation;
      }
      
      // Check for race condition where WooCommerce creates refund records before our process
      $existing_refunds = $order->get_refunds();
      $recent_wc_refund = null;
      
      foreach ($existing_refunds as $refund) {
         $refund_age = time() - $refund->get_date_created()->getTimestamp();
         $refund_amount = wc_format_decimal($refund->get_amount(), wc_get_price_decimals());
         $expected_amount = wc_format_decimal($amount, wc_get_price_decimals());
         
         if ($refund_age < 5 && $refund_amount == $expected_amount) {
            $recent_wc_refund = $refund;
            break;
         }
      }



      $item_total_amount = 0;
      foreach ($order->get_items() as $item) {
         $item_total_amount += $item->get_total() + $item->get_total_tax();
      }

      $current_tax = 0;
      $current_shipping = 0;
      $current_shipping_tax = 0;
      $current_cart = 0;
      $prev_tax = 0;
      $prev_shipping = 0;
      $prev_shipping_tax = 0;
      $prev_cart = 0;
      $totalAmountWithOutFees = 0;
      foreach ($order->get_refunds() as $ind => $item) {
         if ($ind == 0) {
            $current_tax = (-1 * $item->get_cart_tax()); //cart_tax
            $current_shipping = (-1 * $item->get_shipping_total()); //shipping_total
            $current_shipping_tax = (-1 * $item->get_shipping_tax()); //shipping_tax
            $current_cart = (-1 * $item->get_total()); //amount
            $totalAmountWithOutFees = $current_cart;
            $current_cart = $current_cart - $current_tax - $current_shipping - $current_shipping_tax;
         } else {
            $prev_tax += (-1 * $item->get_cart_tax());
            $prev_shipping += (-1 * $item->get_shipping_total());
            $prev_shipping_tax += (-1 * $item->get_shipping_tax());
            $prev_cart += (-1 * $item->get_total());
            $totalAmountWithOutFees += $prev_cart;
            $prev_cart = $prev_cart - $prev_tax - $prev_shipping - $prev_shipping_tax;
         }
      }

      $fully_refund = false;
      $order_fee_total = 0;
      $qty_refunded = 0;
      $prod_items = array();
      $amt = 0;
      $refundTotalAmt = 0;
      $refundTaxAmt = 0;


      foreach ($order->get_items() as $item_id => $item) {
         $prod_items[] = $item_id;
         $qty = $order->get_qty_refunded_for_item($item_id);

         $refundTotalAmt += $order->get_total_refunded_for_item($item_id);
         $qty_refunded += $qty;
      }

      foreach ($line_item_totals as $ind => $val) {
         if (in_array($ind, $prod_items))
            $amt += $val;
      }

      foreach ($line_item_tax_totals as $ind => $values) {
         if (in_array($ind, $prod_items)) {
            foreach ($values as $i => $val) {
               $amt += $val;
            }
         }
      }

      $total_items_price_excluding_tax = 0;
      foreach ($line_item_totals  as $key => $value) {
         $total_items_price_excluding_tax += $value;
      }

      $feeItems = $order->get_items(array('fee'));

      $chargeanywhere_feeitems_mapped = array();
      $refund_fee_items = array();
      foreach ($feeItems as $item_id => $item) {
         $item_data = $item->get_data();

         // Get fee type from metadata (new orders) or fall back to label matching (legacy orders)
         $fee_type = $item->get_meta('_fee_type', true);
         
         if (!empty($fee_type)) {
            // Use metadata-based mapping (more reliable)
            $chargeanywhere_feeitems_mapped[$fee_type]['id'] = $item_data['id'];
            $chargeanywhere_feeitems_mapped[$fee_type]['amount'] = $item_data['total'];
         } else {
            // Fallback to label-based mapping for legacy orders
            if (mb_strtolower($options['credit_service_fee']) == "")
            $options['credit_service_fee'] = 'service fee';
            if (mb_strtolower($options['credit_convenience_service_fee']) == "")
            $options['credit_convenience_service_fee'] = 'convenience fee';
            if (mb_strtolower($options['ach_service_fee']) == "")
            $options['ach_service_fee'] = 'service fee';
            if (mb_strtolower($options['ach_convenience_service_fee']) == "")
            $options['ach_convenience_service_fee'] = 'convenience fee';

         if ($tran_meta['chargeanywhere_option'] == 'credit') {
               if (mb_strtolower(trim($options['credit_service_fee'])) == mb_strtolower(trim($item_data['name']))) {
               $chargeanywhere_feeitems_mapped['credit_service_fee']['id'] = $item_data['id'];
               $chargeanywhere_feeitems_mapped['credit_service_fee']['amount'] = $item_data['total'];
            }
               if (mb_strtolower(trim($options['credit_convenience_service_fee'])) == mb_strtolower(trim($item_data['name']))) {
               $chargeanywhere_feeitems_mapped['credit_convenience_service_fee']['id'] = $item_data['id'];
               $chargeanywhere_feeitems_mapped['credit_convenience_service_fee']['amount'] = $item_data['total'];
            }
         } else {
               if (mb_strtolower(trim($options['ach_service_fee'])) == mb_strtolower(trim($item_data['name']))) {
               $chargeanywhere_feeitems_mapped['ach_service_fee']['id'] = $item_data['id'];
               $chargeanywhere_feeitems_mapped['ach_service_fee']['amount'] = $item_data['total'];
            }
               if (mb_strtolower(trim($options['ach_convenience_service_fee'])) == mb_strtolower(trim($item_data['name']))) {
               $chargeanywhere_feeitems_mapped['ach_convenience_service_fee']['id'] = $item_data['id'];
               $chargeanywhere_feeitems_mapped['ach_convenience_service_fee']['amount'] = $item_data['total'];
               }
            }
         }
      }

      foreach ($order->get_fees() as $fee_id => $fee) {
         $order_fee_total += $fee->get_total();
      }
      $total = $order->get_total();

      $order_tax = $current_shipping_tax + $current_tax;

      $order_ship = $current_shipping;

      $inc = 0;
      $service_amount = 0;

      // Credit service fee pro-rata refund with tax-aware calculation
      if ($tran_meta['chargeanywhere_option'] == 'credit' && $options['credit_refund_service_fee'] == 'yes') {
         if ($item_total_amount > 0 && isset($chargeanywhere_feeitems_mapped['credit_service_fee'])) {
            $fee_data = $chargeanywhere_feeitems_mapped['credit_service_fee'];
            if (isset($fee_data['id']) && isset($fee_data['amount'])) {
               // Check if fee was calculated with tax inclusion (from stored metadata or fallback to current settings)
               $fee_item = null;
               foreach ($feeItems as $item_id => $item) {
                  if ($item_id == $fee_data['id']) {
                     $fee_item = $item;
                     break;
                  }
               }
               
               $tax_included_in_original = 'no';
               if ($fee_item) {
                  $tax_included_in_original = $fee_item->get_meta('_tax_included_in_calculation', true);
                  if (empty($tax_included_in_original)) {
                     // Fallback to current settings for legacy orders
                     $tax_included_in_original = $options['apply_credit_tax_amount_service_fee'];
                  }
               }
               
               // Calculate refund using same logic as original fee calculation
               if ($tax_included_in_original == 'yes') {
                  // Original fee included tax, so use total amount including tax for pro-rata
                  $service_amount = (float) wc_format_decimal((($amt / $item_total_amount) * $fee_data['amount']), wc_get_price_decimals());
               } else {
                  // Original fee excluded tax, use simpler pro-rata
                  $service_amount = (float) wc_format_decimal((($amt / $item_total_amount) * $fee_data['amount']), wc_get_price_decimals());
               }
               
               if ($service_amount > 0) {
                  $refund_fee_items[$inc]['id'] = $fee_data['id'];
                  $refund_fee_items[$inc]['amount'] = $service_amount;
         $inc++;
      }
            }
         }
      }
      
      // ACH service fee pro-rata refund with tax-aware calculation
      if ($tran_meta['chargeanywhere_option'] == 'ach' && $options['ach_refund_service_fee'] == 'yes') {
         if ($item_total_amount > 0 && isset($chargeanywhere_feeitems_mapped['ach_service_fee'])) {
            $fee_data = $chargeanywhere_feeitems_mapped['ach_service_fee'];
            if (isset($fee_data['id']) && isset($fee_data['amount'])) {
               // Check if fee was calculated with tax inclusion (from stored metadata or fallback to current settings)
               $fee_item = null;
               foreach ($feeItems as $item_id => $item) {
                  if ($item_id == $fee_data['id']) {
                     $fee_item = $item;
                     break;
                  }
               }
               
               $tax_included_in_original = 'no';
               if ($fee_item) {
                  $tax_included_in_original = $fee_item->get_meta('_tax_included_in_calculation', true);
                  if (empty($tax_included_in_original)) {
                     // Fallback to current settings for legacy orders
                     $tax_included_in_original = $options['apply_ach_tax_amount_service_fee'];
                  }
               }
               
               // Calculate refund using same logic as original fee calculation
               if ($tax_included_in_original == 'yes') {
                  // Original fee included tax, so use total amount including tax for pro-rata
                  $service_amount = (float) wc_format_decimal((($amt / $item_total_amount) * $fee_data['amount']), wc_get_price_decimals());
               } else {
                  // Original fee excluded tax, use simpler pro-rata
                  $service_amount = (float) wc_format_decimal((($amt / $item_total_amount) * $fee_data['amount']), wc_get_price_decimals());
               }
               
               if ($service_amount > 0) {
                  $refund_fee_items[$inc]['id'] = $fee_data['id'];
                  $refund_fee_items[$inc]['amount'] = $service_amount;
         $inc++;
               }
            }
         }
      }

      $totalRefund = 0;
      $totalRefundTax = 0;
      foreach ($order->get_refunds() as $item) {
         // Check both new metadata (for new refunds) and legacy reason string (for existing refunds)
         if ($item->get_meta('_caw_adjustment_refund', true) == '1' || $item->get_reason() == 'Adjustment to close the order') {
            $totalRefundTax += $item->get_total();
         } else {
            $totalRefund += $item->get_total();
         }
      }

      $total_quantity = $order->get_item_count();
      $qty_refunded = $qty_refunded * (-1);

      $conAmt = 0;
      if ($total_quantity == $qty_refunded) {
         // Full refund: adjust service fee for previous tax refunds
         if ($tran_meta['chargeanywhere_option'] == 'credit' && $options['credit_refund_service_fee'] == 'yes') {
            if ($service_amount > 0 && isset($chargeanywhere_feeitems_mapped['credit_service_fee']['amount']) && $inc > 0) {
               $original_fee_amount = wc_format_decimal($chargeanywhere_feeitems_mapped['credit_service_fee']['amount'], wc_get_price_decimals());
               if ($original_fee_amount <= ($service_amount + ($totalRefundTax * -1))) {
                  $refund_fee_items[$inc - 1]['amount'] = wc_format_decimal($original_fee_amount - ($totalRefundTax * -1), wc_get_price_decimals());
               }
            }
         }

         // Full refund: credit convenience fee
         if ($tran_meta['chargeanywhere_option'] == 'credit' && $options['credit_refund_convenience_fee'] == 'yes') {
            // Try both new key format (credit_convenience_fee) and legacy key format (credit_convenience_service_fee)
            $convenience_fee_key = isset($chargeanywhere_feeitems_mapped['credit_convenience_fee']) ? 'credit_convenience_fee' : 'credit_convenience_service_fee';
            if (isset($chargeanywhere_feeitems_mapped[$convenience_fee_key]['id']) && isset($chargeanywhere_feeitems_mapped[$convenience_fee_key]['amount'])) {
               if (isset($chargeanywherefee_details['credit_convenience_service_fee'])) {
            $service_amount += $chargeanywherefee_details['credit_convenience_service_fee'];
               }
               $refund_fee_items[$inc]['id'] = $chargeanywhere_feeitems_mapped[$convenience_fee_key]['id'];
               $refund_fee_items[$inc]['amount'] = wc_format_decimal($chargeanywhere_feeitems_mapped[$convenience_fee_key]['amount'], wc_get_price_decimals());
            $conAmt = $refund_fee_items[$inc]['amount'];
               $inc++;
            }
         }

         // Full refund: adjust ACH service fee for previous tax refunds
         if ($tran_meta['chargeanywhere_option'] == 'ach' && $options['ach_refund_service_fee'] == 'yes') {
            if ($service_amount > 0 && isset($chargeanywhere_feeitems_mapped['ach_service_fee']['amount']) && $inc > 0) {
               $original_fee_amount = wc_format_decimal($chargeanywhere_feeitems_mapped['ach_service_fee']['amount'], wc_get_price_decimals());
               if ($original_fee_amount <= ($service_amount + ($totalRefundTax * -1))) {
                  $refund_fee_items[$inc - 1]['amount'] = wc_format_decimal($original_fee_amount - ($totalRefundTax * -1), wc_get_price_decimals());
               }
            }
         }

         // Full refund: ACH convenience fee
         if ($tran_meta['chargeanywhere_option'] == 'ach' && $options['ach_refund_convenience_fee'] == 'yes') {
            // Try both new key format (ach_convenience_fee) and legacy key format (ach_convenience_service_fee)
            $convenience_fee_key = isset($chargeanywhere_feeitems_mapped['ach_convenience_fee']) ? 'ach_convenience_fee' : 'ach_convenience_service_fee';
            if (isset($chargeanywhere_feeitems_mapped[$convenience_fee_key]['id']) && isset($chargeanywhere_feeitems_mapped[$convenience_fee_key]['amount'])) {
               if (isset($chargeanywherefee_details['ach_convenience_service_fee'])) {
            $service_amount += $chargeanywherefee_details['ach_convenience_service_fee'];
               }
               $refund_fee_items[$inc]['id'] = $chargeanywhere_feeitems_mapped[$convenience_fee_key]['id'];
               $refund_fee_items[$inc]['amount'] = wc_format_decimal($chargeanywhere_feeitems_mapped[$convenience_fee_key]['amount'], wc_get_price_decimals());
            $conAmt = $refund_fee_items[$inc]['amount'];
               $inc++;
            }
         }
         $fully_refund = true;
      }

      if ($amount > 0) {
         try {
            // Since WooCommerce now includes fees in the main refund amount, pass service_amount as 0
            $response = $this->refund($this, $order, $amount, 0, array("tax" => $order_tax, "shipping" => $order_ship));

            if (isset($response['ResponseCode']) && '000' == $response['ResponseCode']) {
               // Handle existing WooCommerce-created refund (race condition)
               $expected_total_amount = wc_format_decimal($amount, wc_get_price_decimals());
               $existing_main_refund = null;
               
               if ($recent_wc_refund) {
                  $existing_main_refund = $recent_wc_refund;
               } else {
                  $existing_main_refund = $this->find_matching_wc_refund($order, $expected_total_amount);
               }
               
               if ($existing_main_refund) {
                  $existing_main_refund->update_meta_data('_chargeanywhere_refund_reference', isset($response['ReferenceNumber']) ? $response['ReferenceNumber'] : '');
                  $existing_main_refund->update_meta_data('_chargeanywhere_response_code', $response['ResponseCode']);
                  $existing_main_refund->set_reason(sprintf(__('Charge Anywhere refund - Reference: %s', 'woocommerce-chargeanywhere'), isset($response['ReferenceNumber']) ? $response['ReferenceNumber'] : 'N/A'));
                  $existing_main_refund->save();
               }
               
               $refunded_amount = wc_format_decimal($amount, wc_get_price_decimals());
               $line_items = array();
               $total_line_item_amount = 0;
               
               if (count($refund_fee_items) > 0) {
                  foreach ($refund_fee_items as $key => $value) {
                     // Ensure we have valid item data
                     if (!isset($value['id']) || !isset($value['amount']) || $value['amount'] <= 0) {
                        continue;
                     }
                     
                     // Get the fee item to validate it exists
                     $fee_item = null;
                     foreach ($order->get_items('fee') as $item_id => $item) {
                        if ($item_id == $value['id']) {
                           $fee_item = $item;
                           break;
                        }
                     }
                     
                     if (!$fee_item) {
                        continue; // Skip if fee item not found
                     }
                     
                     // Create proper line items structure for WooCommerce
                     // Format amount properly for WooCommerce
                     $refund_amount = wc_format_decimal($value['amount'], wc_get_price_decimals());
                     
                     $line_items[$value['id']] = array(
                        'qty' => 0,
                        'refund_total' => $refund_amount,
                        'refund_tax' => array(),
                        'total' => $refund_amount  // Add missing 'total' key that WooCommerce expects
                     );
                     $total_line_item_amount += $value['amount'];
                  }

               }
               $refunded_amount += $total_line_item_amount;
               if ($fully_refund) {
                  $order = wc_get_order($order_id);
                  $tran_meta = $order->get_meta('_chargeanywhere_transaction');
                  $tran_meta['transaction_type'] = 'complete';
                  $order->update_meta_data('_chargeanywhere_transaction', $tran_meta);
                  $order->save();

                  if ($order->get_total() != wc_format_decimal($totalAmountWithOutFees + $conAmt, wc_get_price_decimals()))
                     $order->update_status('refunded', __('Order fully refunded via Charge Anywhere.', 'woocommerce-cardpay-chargeanywhere'));
               }

               $order->add_order_note(sprintf(__('Refund completed: %s%s (ID: %s)', 'woocommerce-chargeanywhere'), 
                  get_woocommerce_currency_symbol(), number_format($refunded_amount, 2), $response['ReferenceNumber']
               ));
               
               $this->cleanup_refund_protection();
               return true;
            } else {
               $error_text = isset($response['ResponseText']) ? $response['ResponseText'] : 'Refund failed';
               
               $this->cleanup_refund_protection();
               throw new Exception(sprintf(__('Refund failed: %s', 'woocommerce-chargeanywhere'), $error_text));
            }
         } catch (Exception $e) {
            $this->cleanup_refund_protection();
            return new WP_Error('chargeanywhere_error', $e->getMessage());
         }
      } else {
         $this->cleanup_refund_protection();
         return false;
      }
   }

   /**
    * Validate refund request before processing
    * 
    * @param WC_Order $order
    * @param float $amount
    * @param string $reason
    * @return bool|WP_Error
    */
   private function validate_refund_request($order, $amount, $reason)
   {
      // Check if order has Charge Anywhere transaction data
      $tran_meta = $order->get_meta('_chargeanywhere_transaction');
      if (empty($tran_meta) || !is_array($tran_meta)) {
         return new WP_Error('invalid_transaction', __('Order does not have valid Charge Anywhere transaction data', 'woocommerce-chargeanywhere'));
      }
      
      // Check if amount is valid
      if ($amount <= 0) {
         return new WP_Error('invalid_amount', __('Refund amount must be greater than zero', 'woocommerce-chargeanywhere'));
      }
      
      // Check if transaction is auth-only and not yet captured
      $transaction_type = isset($tran_meta['transaction_type']) ? trim($tran_meta['transaction_type']) : '';
      if ($transaction_type === 'authorize') {
         return new WP_Error('auth_only_transaction', __('Cannot refund an authorization-only transaction that has not been captured yet. Please capture the transaction first or void the authorization instead.', 'woocommerce-chargeanywhere'));
      }
      
      // Skip simple total validation - the sophisticated prevent_double_refund() function 
      // handles proper item-level validation with race condition handling
      
      return true;
   }
   

   

   
   /**
    * Prevent double refunding of items
    * 
    * @param WC_Order $order
    * @param array $line_item_totals
    * @param float $amount
    * @return bool|WP_Error
    */
   private function prevent_double_refund($order, $line_item_totals, $amount)
   {
      try {
         $order_id = $order->get_id();
         $existing_refunds = $order->get_refunds();
         
         // Check each line item to ensure we're not refunding more than available
         foreach ($line_item_totals as $item_id => $refund_total) {
            if ($refund_total <= 0) continue; // Skip items not being refunded
            
            $item = $order->get_item($item_id);
            if (!$item) continue;
            
            // Calculate refunds manually to avoid WooCommerce caching/timing issues
            $item_total = (float) $item->get_total();
            $manually_calculated_refunded = 0;
            
            // Calculate existing refunds for this item, excluding very recent ones (race condition handling)
            foreach ($existing_refunds as $existing_refund) {
               $refund_age = time() - $existing_refund->get_date_created()->getTimestamp();
               if ($refund_age < 5) continue; // Skip very recent refunds (race condition)
               
               $refund_items = $existing_refund->get_items();
               foreach ($refund_items as $refund_item) {
                  $refunded_item_id = $refund_item->get_meta('_refunded_item_id', true);
                  if ($refunded_item_id == $item_id) {
                     $manually_calculated_refunded += abs($refund_item->get_total());
                  }
               }
            }
            
            $already_refunded = abs($manually_calculated_refunded);
            $remaining_refundable = $item_total - $already_refunded;
            
            // Check if item is already fully refunded
            if ($already_refunded >= $item_total) {
               return new WP_Error('item_fully_refunded', 
                  sprintf(__('Item "%s" has already been fully refunded (%s%s)', 'woocommerce-chargeanywhere'), 
                     $item->get_name(), get_woocommerce_currency_symbol(), number_format($already_refunded, 2)
                  )
               );
            }
            
            // Check if attempting to refund more than available
            $tolerance = 0.02; // Tolerance for rounding errors in pro-rata calculations
            if (($refund_total - $remaining_refundable) > $tolerance) {
               return new WP_Error('double_refund_attempt', 
                  sprintf(__('Cannot refund %s%s for item "%s". Only %s%s available', 'woocommerce-chargeanywhere'), 
                     get_woocommerce_currency_symbol(), number_format($refund_total, 2), 
                     $item->get_name(),
                     get_woocommerce_currency_symbol(), number_format($remaining_refundable, 2)
                  )
               );
            }
         }
         
         return true;
         
      } catch (Exception $e) {
         return new WP_Error('validation_error', __('Refund validation failed', 'woocommerce-chargeanywhere'));
      }
   }
   
   /**
    * Manually calculate refunded amount for a specific item in this order only
    * This avoids WooCommerce caching issues that might pull data from other orders
    * 
    * @param WC_Order $order
    * @param int $item_id
    * @return float
    */
   private function calculate_item_refunds_manually($order, $item_id)
   {
      $total_refunded = 0;
      
      // Get all refunds for this specific order only
      $refunds = $order->get_refunds();
      
      foreach ($refunds as $refund) {
         $refund_items = $refund->get_items();
         
         foreach ($refund_items as $refund_item_id => $refund_item) {
            // Check if this refund item corresponds to our original item
            $refunded_item_id = $refund_item->get_meta('_refunded_item_id', true);
            
            if ($refunded_item_id == $item_id) {
               $total_refunded += abs($refund_item->get_total());
            }
         }
      }
      
      return $total_refunded;
   }
   
   /**
    * Find an existing WooCommerce refund that matches our expected amount
    * This handles cases where WC creates refund records before our process runs
    * 
    * @param WC_Order $order
    * @param float $expected_amount
    * @return WC_Order_Refund|null
    */
   private function find_matching_wc_refund($order, $expected_amount)
   {
      try {
         $refunds = $order->get_refunds();
         
         if (empty($refunds)) {
            return null;
         }
         
         $expected_amount = wc_format_decimal($expected_amount, wc_get_price_decimals());
         
         foreach ($refunds as $refund) {
            if (!$refund || !is_object($refund)) {
               continue;
            }
            
            $refund_amount = wc_format_decimal($refund->get_amount(), wc_get_price_decimals());
            
            // Check if amounts match and if it lacks CAW metadata (indicating WC created it)
            if ($refund_amount == $expected_amount) {
               $has_caw_metadata = $refund->get_meta('_chargeanywhere_refund_reference', true) || 
                                  $refund->get_meta('_chargeanywhere_response_code', true) ||
                                  $refund->get_meta('_caw_adjustment_refund', true);
               
               // For race condition handling, also check if this is a very recent refund
               $refund_age = time() - $refund->get_date_created()->getTimestamp();
               
               if (!$has_caw_metadata || $refund_age < 10) {
                  // Either no CAW metadata (WC created) or very recent (potential race condition)
                  return $refund;
               }
            }
         }
         
         return null;
      } catch (Exception $e) {
         return null;
      }
      }
   
   /**
    * Check for previous successful API refunds to prevent "Multiple Returns not allowed" errors
    * Charge Anywhere gateway does not allow multiple refund API calls to the same transaction
    * 
    * @param WC_Order $order
    * @return bool|WP_Error
    */
   private function check_previous_api_refunds($order)
   {
      // Method 1: Check order notes for previous successful API responses
      $order_notes = wc_get_order_notes(array(
         'order_id' => $order->get_id(),
         'limit' => 20,
         'orderby' => 'date_created',
         'order' => 'DESC'
      ));
      
      foreach ($order_notes as $note) {
         // Look for successful API response notes
         if (strpos($note->content, 'Charge Anywhere API refund successful') !== false) {
            // Extract reference number
            preg_match('/Reference: (\w+)/', $note->content, $matches);
            $reference = isset($matches[1]) ? $matches[1] : 'Unknown';
            
            $refund_time = $note->date_created->format('Y-m-d H:i:s');
            $currency = get_woocommerce_currency_symbol();
            
            return new WP_Error('multiple_returns_blocked', 
               sprintf(__('Multiple refunds not allowed. Previous refund: %s (%s)', 'woocommerce-chargeanywhere'), 
                  $reference, $refund_time
               )
            );
         }
      }
      
      // Method 2: Check refund records with CAW API reference metadata
      $refunds = $order->get_refunds();
      foreach ($refunds as $refund) {
         $api_reference = $refund->get_meta('_chargeanywhere_refund_reference', true);
         $response_code = $refund->get_meta('_chargeanywhere_response_code', true);
         
         // If we find a refund with API metadata indicating successful processing
         if (!empty($api_reference) && $response_code === '000') {
            $refund_time = $refund->get_date_created()->format('Y-m-d H:i:s');
            
            return new WP_Error('multiple_returns_blocked',
               sprintf(__('Multiple refunds not allowed. Previous refund: %s (%s)', 'woocommerce-chargeanywhere'),
                  $api_reference, $refund_time
               )
            );
         }
      }
      
      return true;
   }
   
   /**
    * Check for recent refund attempts to prevent duplicate API calls
    * 
    * @param WC_Order $order
    * @param float $amount
    * @return bool|WP_Error
    */
   private function check_recent_refund_attempt($order, $amount)
   {
      // Check if a refund with the same amount was processed recently (within last 60 seconds)
      $recent_refunds = $order->get_refunds();
      $current_time = time();
      
      foreach ($recent_refunds as $refund) {
         $refund_time = $refund->get_date_created()->getTimestamp();
         $time_diff = $current_time - $refund_time;
         
         // Check if refund was created within last 60 seconds and has same amount
         if ($time_diff < 60 && abs($refund->get_amount() - $amount) < 0.01) {
            return new WP_Error('duplicate_refund', 
               sprintf(__('A refund for %s%s was already processed %d seconds ago. Please wait before attempting another refund.', 'woocommerce-chargeanywhere'), 
                  get_woocommerce_currency_symbol(), number_format($amount, 2),
                  $time_diff
               )
            );
         }
      }
      
      // Check for ongoing refund process (using transient)
      $refund_lock_key = 'caw_refund_lock_' . $order->get_id();
      if (get_transient($refund_lock_key)) {
         return new WP_Error('refund_in_progress', __('Another refund is currently being processed for this order. Please wait.', 'woocommerce-chargeanywhere'));
      }
      
      // Set lock for this refund process (expires in 2 minutes)
      set_transient($refund_lock_key, $amount, 120);
      
      return true;
   }

   /**
    * Protect refund process against WordPress/WooCommerce interference
    * 
    * @param int $order_id
    */
   private function protect_refund_process($order_id)
   {
      // Minimal protection - just ensure consistent decimal precision
      add_filter('wc_get_price_decimals', array($this, 'ensure_consistent_decimals'), 999);
      
      // Store order ID for reference
      $this->current_refund_order_id = $order_id;
   }
   
   /**
    * Prevent WooCommerce from automatically refunding fees we handle manually
    */
   public function prevent_automatic_fee_refunds($meta, $item, $refund, $order)
   {
      if ($item->get_type() === 'fee') {
         $fee_type = $item->get_meta('_fee_type', true);
         // If this is one of our managed fee types, don't let WC auto-refund it
         if (in_array($fee_type, ['credit_service_fee', 'credit_convenience_fee', 'ach_service_fee', 'ach_convenience_fee'])) {
            return false; // Prevent automatic refund creation
         }
      }
      return $meta;
   }
   
   /**
    * Additional protection: prevent our fee items from being included in regular refunds
    */
   public function prevent_fee_line_items_in_refunds($refund_item, $item_id, $refund_amount, $refund_tax, $refund)
   {
      if (!empty($this->current_refund_order_id)) {
         $order = wc_get_order($this->current_refund_order_id);
         if ($order) {
            $items = $order->get_items('fee');
            if (isset($items[$item_id])) {
               $fee_item = $items[$item_id];
               $fee_type = $fee_item->get_meta('_fee_type', true);
               if (in_array($fee_type, ['credit_service_fee', 'credit_convenience_fee', 'ach_service_fee', 'ach_convenience_fee'])) {
                  // Check if this is our fee adjustment refund (allow it)
                  if ($refund->get_reason() === 'Fee adjustment' || $refund->get_meta('_caw_adjustment_refund', true) == '1') {
                     return $refund_item;
                  }
                  // Otherwise prevent it
                  return false; // Block fee from being included in regular refunds
               }
            }
         }
      }
      return $refund_item;
   }
   
   /**
    * Preserve fee tax status during refunds
    */
   public function preserve_fee_tax_status($taxes, $item)
   {
      if ($item->get_type() === 'fee') {
         $fee_type = $item->get_meta('_fee_type', true);
         if (in_array($fee_type, ['credit_service_fee', 'credit_convenience_fee', 'ach_service_fee', 'ach_convenience_fee'])) {
            // Our fees are always tax-free, don't let WC recalculate
            return array();
         }
      }
      return $taxes;
   }
   
   /**
    * Protect our fee refund amounts from third-party modifications
    */
   public function protect_fee_refund_amounts($refund_item, $item_id, $refund_amount, $refund_tax, $refund)
   {
      if (!empty($this->current_refund_order_id)) {
         $order = wc_get_order($this->current_refund_order_id);
         if ($order) {
            $items = $order->get_items('fee');
            if (isset($items[$item_id])) {
               $fee_item = $items[$item_id];
               $fee_type = $fee_item->get_meta('_fee_type', true);
               if (in_array($fee_type, ['credit_service_fee', 'credit_convenience_fee', 'ach_service_fee', 'ach_convenience_fee'])) {
                  // This is our managed fee - don't let other plugins modify the refund amount
                  // We'll handle the refund amount calculation ourselves
               }
            }
         }
      }
      return $refund_item;
   }
   
   /**
    * Ensure consistent decimal precision for our calculations
    */
   public function ensure_consistent_decimals($decimals)
   {
      // Force consistent decimals during our refund process
      return 2; // Always use 2 decimal places for our calculations
   }
   

   /**
    * Clean up protection hooks after refund processing
    */
   private function cleanup_refund_protection()
   {
      remove_filter('wc_get_price_decimals', array($this, 'ensure_consistent_decimals'), 999);
      
      // Clear refund lock transient
      if ($this->current_refund_order_id) {
         $refund_lock_key = 'caw_refund_lock_' . $this->current_refund_order_id;
         delete_transient($refund_lock_key);
      }
      

      
      $this->current_refund_order_id = null;
   }

   /**
    * refund function
    * 
    * @param WC_Order                   $order
    * @param float                      $amount
    * 
    * @return mixed
    */
   public function refund($gateway, $order, $amount, $service_amount, $extra)
   {
      $payload = $this->get_payload($gateway, $order, $amount, $service_amount, $extra, 'refundTransaction');
      if ($payload != false) {
         $response = $this->post_transaction($payload);
         return $response;
      }
      return true;
   }

   /**
    * Admin Charge any where Settings form
    */
   function init_form_fields()
   {
      $this->form_fields = array(
         'enabled'      => array(
            'title'        => __('Enable/Disable', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Enable Charge Anywhere Payment Module.', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'title'        => array(
            'title'        => __('Title:', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'description'  => __('This controls the title which the user sees during checkout.', 'chargeanywhere-woocommerce'),
            'default'      => __('Charge Anywhere Gateway', 'chargeanywhere-woocommerce')
         ),
         'description'  => array(
            'title'        => __('Description:', 'chargeanywhere-woocommerce'),
            'type'         => 'textarea',
            'description'  => __('This controls the description which the user sees during checkout.', 'chargeanywhere-woocommerce'),
            'default'      => __('Pay securely by Credit or Debit Card through Charge Anywhere Secure Servers.', 'chargeanywhere-woocommerce')
         ),
         'merchant_id'     => array(
            'title'        => __('Merchant ID', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'required'     => true,
            'description'  => __('This is Merchant ID')
         ),
         'terminal_id' => array(
            'title'        => __('Terminal ID', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'description'  =>  __('This is Terminal ID', 'chargeanywhere-woocommerce')
         ),
         'secret_key' => array(
            'title'        => __('Secret', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'description'  =>  __('Secret used to Authenticate the request with Charge Anywhere', 'chargeanywhere-woocommerce')
         ),
         'success_message' => array(
            'title'        => __('Transaction Success Message', 'chargeanywhere-woocommerce'),
            'type'         => 'textarea',
            'description' =>  __('Message to be displayed on successful transaction.', 'chargeanywhere-woocommerce'),
            'default'      => __('Your payment has been processed successfully.', 'chargeanywhere-woocommerce')
         ),
         'failed_message'  => array(
            'title'        => __('Transaction Failed Message', 'chargeanywhere-woocommerce'),
            'type'         => 'textarea',
            'description'  =>  __('Message to be displayed on failed transaction.', 'chargeanywhere-woocommerce'),
            'default'      => __('Your transaction has been declined.', 'chargeanywhere-woocommerce')
         ),
         'redirect_message'  => array(
            'title'        => __('Transaction Redirect Message', 'chargeanywhere-woocommerce'),
            'type'         => 'textarea',
            'description'  =>  __('Message to be displayed on payment redirection.', 'chargeanywhere-woocommerce'),
            'default'      => __('Thank you for your order. We are now redirecting you to Charge Anywhere to make payment.', 'chargeanywhere-woocommerce')
         ),
         'working_mode'    => array(
            'title'        => __('API Mode'),
            'type'         => 'select',
            'options'      => array('false' => 'Live/Production Mode', 'true' => 'Sandbox/Developer API Mode'),
            'description'  => "Live or Production / Sandbox Mode (Sandbox Mode requires Sandbox Account API.)"
         ),
         'transaction_mode'    => array(
            'title'        => __('Transaction Mode'),
            'type'         => 'select',
            'options'      => array('auth_capture' => 'Authorize and Capture', 'authorize' => 'Authorize Only'),
            'description'  => "Transaction Mode. If you are not sure what to use set to Authorize and Capture"
         ),
         'auto_complete_orders' => array(
            'title'        => __('Auto Complete Orders', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Automatically mark orders as completed after successful payment (not for Authorize Only)', 'chargeanywhere-woocommerce'),
            'default'      => 'no',
            'description'  => __('When enabled, orders will be marked as completed regardless of product type', 'chargeanywhere-woocommerce')
         ),
         'email_customer'      => array(
            'title'        => __('Email To Customer', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Enable Sending Default Email To Customer From ChargeAnyWhere', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'email_merchant'      => array(
            'title'        => __('Email To Merchant', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Enable Sending Default Email To Merchant From ChargeAnyWhere', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'log_request'      => array(
            'title'        => __('Enable Log request and response', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Enable Log request and response', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'accept_credit'      => array(
            'title'        => __('Accept Credit', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'apply_credit_service'      => array(
            'title'        => __('', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Add Service Fee', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'credit_service_fee'      => array(
            'title'        => __('Service Fee Label', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'label'        => __('Service Fee Label', 'chargeanywhere-woocommerce'),
            'default'      => 'Credit Card Service Fee'
         ),
         'credit_service_fee_value'      => array(
            'title'        => __('Service Fee Percentage (%)', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'label'        => __('Service Fee Percentage', 'chargeanywhere-woocommerce'),
            'default'      => ''
         ),
         'apply_credit_tax_amount_service_fee'      => array(
            'title'        => __('', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Include Tax Amount in Service Fee Calculation', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'credit_refund_service_fee'      => array(
            'title'        => __('', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Refund Service Fee on a Return', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'apply_credit_convenience_service'      => array(
            'title'        => __('', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Add Convenience Fee', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'credit_convenience_service_fee'      => array(
            'title'        => __('Convenience Fee Label', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'label'        => __('Convenience Fee Label', 'chargeanywhere-woocommerce'),
            'default'      => 'Credit Card Convenience Fee'
         ),
         'credit_convenience_service_fee_value'      => array(
            'title'        => __('Convenience Fee Amount ($)', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'label'        => __('Convenience Fee Amount', 'chargeanywhere-woocommerce'),
            'default'      => ''
         ),
         'credit_refund_convenience_fee'      => array(
            'title'        => __('', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Refund Convenience Fee on a Return', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'accept_ach'      => array(
            'title'        => __('Accept ACH', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'apply_ach_service'      => array(
            'title'        => __('', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Add Service fee', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'ach_service_fee'      => array(
            'title'        => __('Service Fee Label', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'label'        => __('Service Fee Label', 'chargeanywhere-woocommerce'),
            'default'      => 'ACH Service Fee'
         ),
         'ach_service_fee_value'      => array(
            'title'        => __('Service Fee Percentage (%)', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'label'        => __('Service Fee Percentage', 'chargeanywhere-woocommerce'),
            'default'      => ''
         ),
         'apply_ach_tax_amount_service_fee'      => array(
            'title'        => __('', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Include Tax Amount in Service Fee Calculation', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'ach_refund_service_fee'      => array(
            'title'        => __('', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Refund Service Fee on a Return', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'apply_ach_convenience_service'      => array(
            'title'        => __('', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Add Convenience Fee', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
         'ach_convenience_service_fee'      => array(
            'title'        => __('Convenience Fee Label', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'label'        => __('Convenience Fee Label', 'chargeanywhere-woocommerce'),
            'default'      => 'ACH Convenience Fee'
         ),
         'ach_convenience_service_fee_value'      => array(
            'title'        => __('Convenience Fee Amount ($)', 'chargeanywhere-woocommerce'),
            'type'         => 'text',
            'label'        => __('Convenience Fee Amount', 'chargeanywhere-woocommerce'),
            'default'      => ''
         ),
         'ach_refund_convenience_fee'      => array(
            'title'        => __('', 'chargeanywhere-woocommerce'),
            'type'         => 'checkbox',
            'label'        => __('Refund Convenience Fee on a Return', 'chargeanywhere-woocommerce'),
            'default'      => 'no'
         ),
      );
   }

   /** 
    * Validation for Merchant ID
    */
   public function validate_merchant_id_field($key, $value)
   {
      if (trim($value) == '') {
         WC_Admin_Settings::add_error(esc_html__('The Merchant Id doesn\'t look like a correct one.'));
      }
      return $value;
   }

   /**
    * Validation for Terminal ID
    */
   public function validate_terminal_id_field($key, $value)
   {
      if (trim($value) == '') {
         WC_Admin_Settings::add_error(esc_html__('The Terminal Id doesn\'t look like a correct one.'));
      }
      return $value;
   }

   /**
    * Validation for Secret Key
    */
   public function validate_secret_key_field($key, $value)
   {
      if (trim($value) == '') {
         WC_Admin_Settings::add_error(esc_html__('The Secret Key doesn\'t look like a correct one.'));
      }
      return $value;
   }

   public function validate_credit_service_fee_field($key, $value)
   {
      if (preg_match("/tax/i", $value)) {
         WC_Admin_Settings::add_error(esc_html__('Credit Service Fee Label should not contain word tax'));
         $value = ''; // empty it because it is not correct
      }
      return $value;
   }

   public function validate_credit_convenience_service_fee_field($key, $value)
   {
      if (preg_match("/tax/i", $value)) {
         WC_Admin_Settings::add_error(esc_html__('Credit Convenience Service Fee Label should not contain word tax'));
         $value = ''; // empty it because it is not correct
      }
      return $value;
   }

   public function validate_ach_service_fee_field($key, $value)
   {
      if (preg_match("/tax/i", $value)) {
         WC_Admin_Settings::add_error(esc_html__('ACH Service Fee Label should not contain word tax'));
         $value = ''; // empty it because it is not correct
      }
      return $value;
   }

   public function validate_ach_convenience_service_fee_field($key, $value)
   {
      if (preg_match("/tax/i", $value)) {
         WC_Admin_Settings::add_error(esc_html__('ACH Convenience Service Fee Label should not contain word tax'));
         $value = ''; // empty it because it is not correct
      }
      return $value;
   }



   /**
    * Admin Panel Options
    * - Options for bits like 'title' and availability on a country-by-country basis
    **/
   public function admin_options()
   {
      echo '<h3>' . __('Charge Anywhere Payment Gateway', 'chargeanywhere-woocommerce') . '</h3>';
      echo '<table class="form-table">';
      $this->generate_settings_html();
      echo '</table>';
   }

   /**
    *  There are no payment fields for ChargeAnyWhere, but want to show the description if set.
    **/
   function payment_fields()
   {
      if (isset($_GET['ca-error'])) {
         wc_add_notice(__($_GET['ca-error'], 'woothemes'), 'error');
      }

      if ($this->description) {
        
         echo wpautop(wptexturize(__($this->description, 'chargeanywhere-woocommerce')));
       
      }
      include('credit_card.php');
   }

   /**
    * Override payment complete status to prevent auto-completion in authorize mode
    *
    * @param string $status
    * @param int $order_id
    * @param WC_Order $order
    * @return string
    */
   public function override_payment_complete_status($status, $order_id, $order)
   {
      // Only apply to our gateway orders in authorize mode
      if ($order && $order->get_payment_method() === $this->id && $this->transaction_mode === 'authorize') {
         // Always return processing status for authorize-only orders, regardless of product type
         return 'processing';
      }
      return $status;
   }

   /**
    * process_capture function.
    *
    * @access public
    * @param int $order_id
    * @return bool
    */
   public function process_capture($order_id)
   {
      $order = wc_get_order($order_id);
      
      if (!$order) {
         return false;
      }

      // Return if another payment method was used
      $payment_method = version_compare(WC_VERSION, '3.0.0', '<') ? $order->payment_method : $order->get_payment_method();
      if ($payment_method != $this->id) {
         return;
      }



      $tran_meta = $order->get_meta('_chargeanywhere_transaction');

      $orig_tran_type = $tran_meta['transaction_type'];
      $amount = $order->get_total();
      $order_tax = $order->get_total_tax();

      $order_ship = 0;
      foreach ($order->get_items('shipping') as $ship) {
         $order_ship += $ship->get_total();
      }

      $service_amount = 0;
      foreach ($order->get_fees() as $fee_id => $fee) {
         $service_amount += $fee->get_total();
      }

      if ('authorize' == $orig_tran_type) {
         try {
            $amt = $amount - $service_amount;
            $response = $this->capture($this, $order, $amt, $service_amount, array("tax" => $order_tax, "shipping" => $order_ship));
            if (is_wp_error($response)) {
               throw new Exception($response->get_error_message());
            }

            if (isset($response['ResponseCode']) && '000' == $response['ResponseCode']) {

               $order = wc_get_order($order_id);
               $tran_meta = $order->get_meta('_chargeanywhere_transaction');
               
               // Update transaction type and reference details for proper refund handling
               $tran_meta['transaction_type'] = 'auth_capture';
               
               // Update transaction references only if present in response
               if (isset($response['ReferenceNumber'])) {
                  $tran_meta['transaction_id'] = $response['ReferenceNumber']; // Update to capture transaction ID
               }
               if (isset($response['ApprovalCode'])) {
                  $tran_meta['approval_code'] = $response['ApprovalCode']; // Update to capture approval code
               }
               $tran_meta['authorize_amount'] = $amount; // Update to capture amount
               
               $order->update_meta_data('_chargeanywhere_transaction', $tran_meta);
               $order->save();

               $captured_amount = number_format($amount, '2', '.', '');
               $order->add_order_note(sprintf(__('Charge Anywhere auto capture completed for %s. Capture ID: %s', 'chargeanywhere-woocommerce'), $captured_amount, $response['ReferenceNumber']));
               return true;
            } else {
               throw new Exception(__('Charge Anywhere auto capture failed. Log into your gateway to manually process the capture.', 'chargeanywhere-woocommerce'));
            }
         } catch (Exception $e) {
            $order->add_order_note($e->getMessage());
            return true;
         }
      }
   }

   /**
    * capture function
    * 
    * @param WC_Order                   $order
    * @param float                      $amount
    * 
    * @return mixed
    */
   public function capture($gateway, $order, $amount, $service_amount, $extra)
   {
      $payload = $this->get_payload($gateway, $order, $amount, $service_amount, $extra, 'priorAuthCaptureTransaction');
      if ($payload != false) {
         $response = $this->post_transaction($payload);
         return $response;
      }
      return true;
   }

   /**
    * get_payload function
    * 
    * @param WC_Order                   $order
    * @param float                      $amount
    * @param string                     $transaction_type
    * 
    * @return string
    */
   public function get_payload($gateway, $order, $amount, $service_amount, $extra, $transaction_type)
   {
      $order_number = $order->get_id();
      $tran_meta = $order->get_meta('_chargeanywhere_transaction');
      $vSeed = $this->seed_gen();
      $data = array();

      if ($service_amount < 0)
         $service_amount = -1 * $service_amount;
      if ($transaction_type == 'priorAuthCaptureTransaction') {
         if (trim($tran_meta['transaction_type']) != 'authorize') return false;
         $data = array(
            'CustomerNumber' => wc_clean($order->get_customer_id()),
            'Secret' => $gateway->secret_key,
            'MerchantId' => $gateway->merchant_id,
            'TerminalId' => $gateway->terminal_id,
            'InvoiceNumber' => $order_number,
            'Seed' => $vSeed,
            'Mode' => '2',
            'Version' => '2.0',
            'TransactionType' => 'Force',
            'Tax' => $extra['tax'],
            'Shipping' => $extra['shipping'],
            'OriginalTransactionType' => 'Auth Only',
            'OriginalApprovalCode' => $tran_meta['approval_code'],
            'OriginalReferenceNumber' =>  $tran_meta['transaction_id'],
            'OriginalAmount' => $tran_meta['authorize_amount'],
            'EmailCustomer'  => $this->email_customer,
            'EmailMerchant'  => $this->email_merchant,
            'ServiceFeeLabel' => $tran_meta['fee_label'],
            'ServiceFeeAmount' => wc_clean($service_amount),
            'ServiceFeeProvided' => 1,
            'Amount' => wc_clean($amount + $service_amount),
            'AppVersion' => 'WC-4.2'
         );
      } else {
         if (trim($tran_meta['transaction_type']) != 'auth_capture') return false;
         $data = array(
            'CustomerNumber' => wc_clean($order->get_customer_id()),
            'Secret' => $gateway->secret_key,
            'MerchantId' => $gateway->merchant_id,
            'TerminalId' => $gateway->terminal_id,
            'InvoiceNumber' => $order_number,
            'Seed' => $vSeed,
            'Mode' => '2',
            'Version' => '2.0',
            'TransactionType' => 'Return',
            'Tax' => $extra['tax'],
            'Shipping' => $extra['shipping'],
            'OriginalTransactionType' => 'Sale',
            'OriginalApprovalCode' => $tran_meta['approval_code'],
            'OriginalReferenceNumber' =>  $tran_meta['transaction_id'],
            'OriginalAmount' => $tran_meta['authorize_amount'],
            'EmailCustomer'  => $this->email_customer,
            'EmailMerchant'  => $this->email_merchant,
            'ServiceFeeLabel' => $tran_meta['fee_label'],
            'ServiceFeeAmount' => wc_clean($service_amount),
            'ServiceFeeProvided' => 1,
            'Amount' => wc_clean($amount + $service_amount),
            'AppVersion' => 'WC-4.2'
         );
      }
      return json_encode($data);
   }

   /**
    * post_transaction function
    * 
    * @param string $payload
    * @param array  $headers
    * 
    * @return string|WP_Error
    */
   public function post_transaction($payload)
   {
      $data = json_decode($payload);
      $vPostData = "";
      foreach ($data as $key => $val) {
         if ($vPostData != "") $vPostData .= "&";
         $vPostData .= $key . "=" . $val;
      }

      $processURI = '';
      if ($this->mode == 'true') {
         $processURI = $this->test_url_process;
      } else {
         $processURI = $this->live_url_process;
      }

      if ($this->log_request == 'true') {
         $data = json_decode(json_encode($data), true);
         $this->logData("Request", array(
            'merchant_id'  => $this->merchant_id,
            'terminal_id'  => $this->terminal_id,
            'invoiceNumber' => $data['InvoiceNumber'],
            'customerNumber' => $data['CustomerNumber'],
            'amount' => $data['Amount'],
            'transactionType' => $data['TransactionType'],
            'ServiceFeeLabel' => $data['ServiceFeeLabel'],
            'ServiceFeeAmount' => $data['ServiceFeeAmount'],
            'OriginalTransactionType' => $data['OriginalTransactionType'],
            'data' => $data,
         ));
      }

      $response = wp_remote_post($processURI, array('method' => 'POST', 'body' => $vPostData));

      if (is_wp_error($response)) {
         $error_message = $response->get_error_message();
         return new WP_Error('chargeanywhere_error', __('There was a problem connecting to the payment gateway.' . $error_message, 'chargeanywhere-woocommerce'));
      }

      if ($response['body'] !== FALSE) {
         $responsedata = explode("&", $response['body']);
         $response = array();
         foreach ($responsedata as $key => $val) {
            $txt = explode("=", $val);
            $response[$txt[0]] = $txt[1];
         }

         if ($this->log_request == 'true') {
            $this->logData("Response",  array(
               'merchant_id'  => $this->merchant_id,
               'terminal_id'  => $this->terminal_id,
               'invoicenumber' => $data['InvoiceNumber'],
               'responsecode' => $response['ResponseCode'],
               'approvalcode' => $response['ApprovalCode'],
               'AuthorizedAmount' => $response['AuthorizedAmount'],
               'ReferenceNumber' => $response['ReferenceNumber'],
               'transactionType' => $data['TransactionType'],
            ));
         }

         return $response;
      } else {
            /*Handle Failure Case*/;
         return new WP_Error('chargeanywhere_error', __('There was a problem connecting to the payment gateway.', 'chargeanywhere-woocommerce'));
      }
   }

   /**
    * Receipt Page
    **/
   function receipt_page($order)
   {
      echo '<p>' . __('Thank you for your order, please click the button below to pay with Charge Anywhere.', 'chargeanywhere-woocommerce') . '</p>';
      echo $this->generate_authorize_form($order);
   }

   /**
    * Process the payment and return the result
    **/
   function process_payment($order_id)
   {
      $order = wc_get_order($order_id);
      
      if (!$order) {
         return array(
            'result' => 'failure',
            'messages' => __('Invalid order ID', 'woocommerce-chargeanywhere')
         );
      }

      $card_raw       = isset($_POST['chargeanywherecardnumber']) ? sanitize_text_field(wp_unslash($_POST['chargeanywherecardnumber'])) : '';
      $card_number    = str_replace(' ', '', $card_raw);
      $exp_raw        = isset($_POST['chargeanywherecardexpiry']) ? sanitize_text_field(wp_unslash($_POST['chargeanywherecardexpiry'])) : '';
      $exp_date_array = explode('/', $exp_raw);
      $exp_month      = trim(isset($exp_date_array[0]) ? $exp_date_array[0] : '');
      $exp_year       = trim(isset($exp_date_array[1]) ? $exp_date_array[1] : '');
      $cvc            = isset($_POST['chargeanywherecardcvc']) ? sanitize_text_field(wp_unslash($_POST['chargeanywherecardcvc'])) : '';

      $chargeanywhere_option            = isset($_POST['chargeanywhereoption']) ? sanitize_text_field(wp_unslash($_POST['chargeanywhereoption'])) : '';
      WC()->session->set('chargeanywhere_option', $chargeanywhere_option);

      $account_number      = isset($_POST['accountnumber']) ? sanitize_text_field(wp_unslash($_POST['accountnumber'])) : '';
      $account_number      = str_replace(' ', '', $account_number);
      $routing_number      = isset($_POST['routingnumber']) ? sanitize_text_field(wp_unslash($_POST['routingnumber'])) : '';
      $routing_number      = str_replace(' ', '', $routing_number);
      $acc_type            = isset($_POST['accounttype']) ? sanitize_text_field(wp_unslash($_POST['accounttype'])) : '';

      return array(
         'result'    => 'success',
         'redirect'   => $order->get_checkout_payment_url(true)
      );
   }

   /**
    * Check for valid chargeanywhere server callback to validate the transaction response.
    **/
   function check_authorize_response()
   {
      global $woocommerce;
      $invoicenumber = isset($_POST['InvoiceNumber']) ? sanitize_text_field(wp_unslash($_POST['InvoiceNumber'])) : '';
      $order = wc_get_order($invoicenumber);
      
      if (!$order) {
         wp_safe_redirect(wc_get_checkout_url() . '?ca-error=' . __('Invalid order', 'woocommerce-chargeanywhere'));
         exit;
      }

      if (count($_POST)) {
         $redirect_url = '';
         $this->msg['class']     = 'error';
         $this->msg['message']   = $this->failed_message;

         //Response Data

         $responsecode = isset($_POST['ResponseCode']) ? sanitize_text_field(wp_unslash($_POST['ResponseCode'])) : '';
         $amount = isset($_POST['Amount']) ? sanitize_text_field(wp_unslash($_POST['Amount'])) : '';
         $approvalcode = isset($_POST['ApprovalCode']) ? sanitize_text_field(wp_unslash($_POST['ApprovalCode'])) : '';
         $authorize_amount = isset($_POST['AuthorizedAmount']) ? sanitize_text_field(wp_unslash($_POST['AuthorizedAmount'])) : '';

         $fee = isset($_POST['fee']) ? sanitize_text_field(wp_unslash($_POST['fee'])) : '';
         $conv_fee = isset($_POST['conv_fee']) ? sanitize_text_field(wp_unslash($_POST['conv_fee'])) : '';

         $card_number = isset($_POST['CardNumber']) ? sanitize_text_field(wp_unslash($_POST['CardNumber'])) : '';
         $reference_number = isset($_POST['ReferenceNumber']) ? sanitize_text_field(wp_unslash($_POST['ReferenceNumber'])) : '';
         $expiry_month = isset($_POST['ExpMonth']) ? sanitize_text_field(wp_unslash($_POST['ExpMonth'])) : '';
         $chargeanywhere_option = isset($_POST['chargeanywhere_option']) ? sanitize_text_field(wp_unslash($_POST['chargeanywhere_option'])) : '';
         $chargeanywhere_fee_details = isset($_POST['chargeanywhere_fee_details']) ? sanitize_text_field(wp_unslash($_POST['chargeanywhere_fee_details'])) : '';
         if ($this->log_request == 'true') {
            $this->logData("Response", array(
               'merchant_id'  => $this->merchant_id,
               'terminal_id'  => $this->terminal_id,
               'invoicenumber' => $invoicenumber,
               'responsecode' => $responsecode,
               'approvalcode' => $approvalcode,
               'amount' => $amount,
               'chargeanywhere_option' => $chargeanywhere_option,
               'transaction_type' => $this->transaction_mode,
            ));
         }
         if ($responsecode != '') {
            try {
               $transauthorised  = false;
               if ($responsecode == '000') {
                     $transauthorised        = true;
                     $this->msg['message']   = $this->success_message;
                     $this->msg['class']     = 'success';

                     // Only process if order hasn't been processed already
                     if ($order->get_status() == 'processing' || $order->get_status() == 'completed') {
                        // Order already processed, just update success message
                     } else {

                        $fee_label = $_POST['ServiceFeeLabel'];


                        if ($fee != 0.00) {

                           $sfee = new WC_Order_Item_Fee();
                           // Use admin's configured label, or fallback if empty
                           $admin_sf_label = ($chargeanywhere_option === 'credit') ? $this->settings['credit_service_fee'] : $this->settings['ach_service_fee'];
                           $payment_type = ($chargeanywhere_option === 'credit') ? 'CC' : 'ACH';
                           if (empty($admin_sf_label)) {
                              $sf_name = "SF_" . $payment_type; // Fallback only if empty
                           } else {
                              $sf_name = $admin_sf_label; // Respect admin's choice
                           }
                           $sfee->set_name($sf_name); // Set the fee name
                           $sfee->set_amount(floatval($fee)); // Set the fee amount
                           $sfee->set_total(floatval($fee)); // Set the total amount of the fee
                           $sfee->set_tax_status('none'); // Set tax status ('taxable', 'none')
                           
                           // Add comprehensive fee metadata for reliable identification and refund handling
                           if ($chargeanywhere_option === 'credit') {
                              $sfee->add_meta_data('_fee_type', 'credit_service_fee', true);
                              $sfee->add_meta_data('_fee_description', 'Credit Card Service Fee', true);
                              $sfee->add_meta_data('_tax_included_in_calculation', $this->settings['apply_credit_tax_amount_service_fee'], true);
                              $sfee->add_meta_data('_fee_percentage', $this->settings['credit_service_fee_value'], true);
                           } else {
                              $sfee->add_meta_data('_fee_type', 'ach_service_fee', true);
                              $sfee->add_meta_data('_fee_description', 'ACH Service Fee', true);
                              $sfee->add_meta_data('_tax_included_in_calculation', $this->settings['apply_ach_tax_amount_service_fee'], true);
                              $sfee->add_meta_data('_fee_percentage', $this->settings['ach_service_fee_value'], true);
                           }
                           
                           $order->add_item($sfee);
                        }

                        if ($conv_fee != 0.00) {

                           $cfee = new WC_Order_Item_Fee();
                           // Use admin's configured label, or fallback if empty
                           $admin_cf_label = ($chargeanywhere_option === 'credit') ? $this->settings['credit_convenience_service_fee'] : $this->settings['ach_convenience_service_fee'];
                           $payment_type = ($chargeanywhere_option === 'credit') ? 'CC' : 'ACH';
                           if (empty($admin_cf_label)) {
                              $cf_name = "CF_" . $payment_type; // Fallback only if empty
                           } else {
                              $cf_name = $admin_cf_label; // Respect admin's choice
                           }
                           $cfee->set_name($cf_name); // Set the fee name
                           $cfee->set_amount(floatval($conv_fee)); // Set the fee amount
                           $cfee->set_total(floatval($conv_fee)); // Set the total amount of the fee
                           $cfee->set_tax_status('none'); // Set tax status ('taxable', 'none')
                           
                           // Add comprehensive fee metadata for reliable identification and refund handling
                           if ($chargeanywhere_option === 'credit') {
                              $cfee->add_meta_data('_fee_type', 'credit_convenience_fee', true);
                              $cfee->add_meta_data('_fee_description', 'Credit Card Convenience Fee', true);
                              $cfee->add_meta_data('_tax_included_in_calculation', 'no', true); // Convenience fees are fixed amounts
                              $cfee->add_meta_data('_fee_amount', $this->settings['credit_convenience_service_fee_value'], true);
                           } else {
                              $cfee->add_meta_data('_fee_type', 'ach_convenience_fee', true);
                              $cfee->add_meta_data('_fee_description', 'ACH Convenience Fee', true);
                              $cfee->add_meta_data('_tax_included_in_calculation', 'no', true); // Convenience fees are fixed amounts
                              $cfee->add_meta_data('_fee_amount', $this->settings['ach_convenience_service_fee_value'], true);
                           }
                           
                           $order->add_item($cfee);
                        }

                        $order->calculate_totals();
                        $order->save();
                        $order->add_order_note('Charge Anywhere payment successful<br/>Ref Number/Transaction ID: ' . $reference_number);
                        $order->add_order_note($this->msg['message']);
                        $trans_id = $reference_number;
                        $tran_meta = array(
                           'transaction_id' => $trans_id,
                           'approval_code' => $approvalcode,
                           'authorize_amount' => $authorize_amount,
                           'cc_last4' => substr($card_number, -4),
                           'cc_expiry' => $expiry_month,
                           'fee_label' => $fee_label,
                           'transaction_type' => $this->transaction_mode,
                           'chargeanywhere_option' => $chargeanywhere_option,
                           'chargeanywhere_fee_details' => $chargeanywhere_fee_details,
                           'fee' => $fee,
                           'conv_fee' => $conv_fee
                        );
                        $order->update_meta_data('_chargeanywhere_transaction', $tran_meta);
                        $order->save();
                        if ($this->transaction_mode == 'authorize') {
                           $order->update_status('Pending Payment');
                        }
                        $woocommerce->cart->empty_cart();
                        $order->payment_complete($invoicenumber);
                        
                        // Auto-complete orders only when transaction mode is auth_capture
                        // Use fallback value if setting is somehow undefined
                        $auto_complete = isset($this->auto_complete_orders) ? $this->auto_complete_orders : 'no';
                        if ($auto_complete === 'yes' && $this->transaction_mode === 'auth_capture') {
                           $order->update_status('completed', __('Order completed after successful payment.', 'woocommerce-chargeanywhere'));
                        }
                     }
                  } else {
                     $this->msg['class'] = 'error';
                     $this->msg['message'] = $this->failed_message;
                     $order->add_order_note($this->msg['message']);
                     $order->update_status('failed');
                     wp_safe_redirect($order->get_checkout_payment_url(true) . '&ca-error=' . $_POST['ResponseText']);
                  }
               
               if ($transauthorised == false) {
                  $order->update_status('failed');
                  $order->add_order_note($this->msg['message']);
               }
            } catch (Exception $e) {
               $msg = "Error";
            }
         } else {
            $this->msg['class'] = 'error';
            $this->msg['message'] = $this->failed_message;
            $order->add_order_note($this->msg['message']);
            $order->update_status('failed');
            wp_safe_redirect(wc_get_checkout_url() . '?ca-error=' . $_POST['ResponseText']);
            exit;
         }
         $redirect_url = $order->get_checkout_order_received_url();
         wp_safe_redirect($redirect_url);
      } else {
         $this->msg['class'] = 'error';
         $this->msg['message'] = $this->failed_message;
         $order->add_order_note($this->msg['message']);
         $order->update_status('failed');
         if (isset($_POST['ResponseText'])) {
            wp_safe_redirect(wc_get_checkout_url() . '?ca-error=' . $_POST['ResponseText']);
         }
      }
   }


   public function web_redirect($url)
   {
      echo "<html><head><script language=\"javascript\">
              <!--
              window.location=\"{$url}\";
              //-->
              </script>
              </head><body><noscript><meta http-equiv=\"refresh\" content=\"0;url={$url}\"></noscript></body></html>";
   }
   /**
    * Generate chargeanywhere button link
    **/
   public function generate_authorize_form($order_id)
   {

      $blocks_support_instance = new WC_Gateway_ChargeAnywhere_Blocks_Support();

      // Call the non-static method on the instance
      $script_handles = $blocks_support_instance->get_payment_method_script_handles();

      // Enqueue the scripts
      foreach ($script_handles as $handle) {
         wp_enqueue_script($handle);
      }
      global $woocommerce;
      $options = get_option('woocommerce_chargeanywhere_settings');
      $order = wc_get_order($order_id);

      $timeStamp     = time();
      $order_total   = $order->get_subtotal()-$order->get_total_discount();  // Get the order total after discount
      $order_total   = number_format((float)$order_total, 2, '.', '');
      $signatureKey  = ($this->secret_key != '') ? $this->secret_key : '';


      $relay_url = get_site_url() . '/wc-api/' . get_class($this);
      $authorize_args = array();
      $authorize_args_array = array();

      $authorize_args['current_time'] = date("Y-m-d H:i:s:u");
      $chargeanywhere_option = WC()->session->get('chargeanywhere_option');
      $fee_store_details['credit_service_fee'] = $options['apply_credit_service'] == 'yes' && $this->settings['credit_service_fee_value'] ? $this->settings['credit_service_fee_value'] : 0;
      $fee_store_details['credit_convenience_service_fee'] = $options['apply_credit_convenience_service'] == 'yes' && $this->settings['credit_convenience_service_fee_value'] ? $this->settings['credit_convenience_service_fee_value'] : 0;
      $fee_store_details['ach_service_fee'] = $options['apply_ach_service'] == 'yes' && $this->settings['ach_service_fee_value'] ? $this->settings['ach_service_fee_value'] : 0;
      $fee_store_details['ach_convenience_service_fee'] = $options['apply_ach_convenience_service'] == 'yes' && $this->settings['ach_convenience_service_fee_value'] ? $this->settings['ach_convenience_service_fee_value'] : 0;
      $credit_fee_label = $options['apply_credit_service'] == 'yes' && $options['credit_service_fee'] ? $options['credit_service_fee'] : $options['credit_convenience_service_fee'];
      $ach_fee_label = $options['apply_ach_service'] == 'yes' && $options['ach_service_fee'] ? $options['ach_service_fee'] : $options['ach_convenience_service_fee'];
      $credit_con_fee_label = $options['credit_convenience_service_fee'];
      $ach_con_fee_label = $options['ach_convenience_service_fee'];



      $authorize_args['TransactionType'] = 'Sale';
      if ($this->transaction_mode == 'authorize')
         $authorize_args['TransactionType'] = 'Auth Only';
      $vSeed = $this->seed_gen();

      $order_tax = 0;
      foreach ($order->get_items('tax') as $tax) {
         $order_tax += $tax->get_tax_total();
      }

      $order_ship = 0;
      $order_ship_tax = 0;
      foreach ($order->get_items('shipping') as $item_id => $shipping_item) {
         $shipping_instance_id = $shipping_item->get_instance_id();

         // Get the shipping method details
         $shipping_method = WC_Shipping_Zones::get_shipping_method($shipping_instance_id);

         // Check if the shipping method is taxable
         if ($shipping_method) {


            // Add shipping tax to total
            $order_ship_tax += $shipping_item->get_total_tax();
         }
         $order_ship += $shipping_item->get_total();
      }



      $order_ship_total = $order_ship_tax + $order_ship;

      $authorize_args['Tax'] = number_format((float)$order_tax, 2, '.', '');
      $authorize_args['Shipping'] = number_format((float)$order_ship_total, 2, '.', '');


      $authorize_args['ServiceFeeProvided'] = 1;

      $authorize_args['MerchantId'] = $this->merchant_id;
      $authorize_args['TerminalId'] = $this->terminal_id;
      $authorize_args['Seed'] = $vSeed;
      $sub_total = max(0, $order_total + $order_tax + $order_ship + $order_ship_tax);
      $authorize_args['Amount'] = number_format((float)$sub_total, 2, '.', '');
      $authorize_args['chargeanywhere_fee_details'] = json_encode($fee_store_details);
      $vTextTohash = $this->merchant_id . ':' . $this->terminal_id . ':' . $vSeed . ':' . number_format((float)$sub_total, 2, '.', '');
      $signature = hash_hmac('sha512', $vTextTohash, hex2bin($this->secret_key));

      $authorize_args['Signature'] = $signature;
      $authorize_args['MerchantReceiptURL'] = $relay_url;
      $authorize_args['Mode'] = '1';
      $authorize_args['Version'] = '2.0';
      $authorize_args['AppVersion'] = 'WC-4.2';
      $authorize_args['InvoiceNumber'] = $order_id;
      $authorize_args['CustomerNumber'] = $order->get_customer_id();
      $authorize_args['BIFirstName'] = $order->get_billing_first_name();
      $authorize_args['BILastName'] = $order->get_billing_last_name();
      $authorize_args['BIAddress'] = $order->get_billing_address_1() . ' ' . $order->get_billing_address_2();
      $authorize_args['BICity'] = $order->get_billing_city();
      $authorize_args['BIState'] = $order->get_billing_state();
      $authorize_args['BIZipCode'] = $order->get_billing_postcode();
      $authorize_args['BICountry'] = $order->get_billing_country();
      $authorize_args['BIEmail'] = $order->get_billing_email();
      $authorize_args['EmailCustomer'] = $this->email_customer;
      $authorize_args['EmailMerchant'] = $this->email_merchant;

      foreach ($authorize_args as $key => $value) {
         $authorize_args_array[] = "<input type='hidden' name='$key' value='$value'/>";
      }

      if (isset($_GET['ca-error'])) {
         echo '<div class="woocommerce-NoticeGroup woocommerce-NoticeGroup-updateOrderReview"> <div class="wc-block-components-notice-banner is-error" role="alert"><div class="wc-block-components-notice-banner__content">' . $_GET['ca-error'] . '</div></div></div>';
      }

      if ($this->mode == 'true') {
         $processURI = $this->test_url_transaction;
         $html_form = '<p><strong>TEST MODE ENABLED:</strong></p> <ul><li>For Credit Card, use test card number 4111111111111111 with any 3-digit CVC and a future expiration date</li>
         <li>For ACH, use test account number 123456 with routing number 111111118</li></ul>';
      } else {
         $processURI = $this->live_url_transaction;
         $html_form = '';
      }
      if ($this->log_request == 'true') {
         $this->logData("Request", array(
            'merchant_id'  => $this->merchant_id,
            'terminal_id'  => $this->terminal_id,
            'invoiceNumber' => $authorize_args['InvoiceNumber'],
            'customerNumber' => $authorize_args['CustomerNumber'],
            'amount' => $authorize_args['Amount'],
            'transactionType' => $authorize_args['TransactionType'],
            'chargeanywhere_option' => $chargeanywhere_option,
         ));
      }



      $credit_service_fee = ($order_total + ($options['apply_credit_tax_amount_service_fee'] == 'yes' ? $order_tax : 0)) * ($fee_store_details['credit_service_fee'] / 100);
      $credit_convenience_service_fee = $fee_store_details['credit_convenience_service_fee'];
      $ach_service_fee = ($order_total + ($options['apply_ach_tax_amount_service_fee'] == 'yes' ? $order_tax : 0)) * ($fee_store_details['ach_service_fee'] / 100);
      $ach_convenience_service_fee = $fee_store_details['ach_convenience_service_fee'];
      $ach_total = $order_total + $order_tax + $order_ship + $order_ship_tax + $ach_service_fee + $ach_convenience_service_fee;
      $credit_total = $order_total + $order_tax + $order_ship + $order_ship_tax + $credit_service_fee + $credit_convenience_service_fee;


      $currency_symbol = get_woocommerce_currency_symbol();

      $html_form .= "<style>
                        table tr td:last-child {
                           text-align: right;
                        } 
                        .error {
                           border: 1px solid red !important;
                        }
						/* CSS for col-md-6 */

						/* Large screens (desktops, min-width: 1200px) */
						@media (min-width: 1200px) {
						  .col-md-6 {
							  width: 49%;
									float: right;
									box-sizing: border-box;
									margin: 0 0.5%;
						  }
						}

						/* Medium screens (tablets, min-width: 992px) */
						@media (min-width: 992px) and (max-width: 1199.98px) {
						  .col-md-6 {
							  width: 49%;
									float: right;
									box-sizing: border-box;
									margin: 0 0.5%;
						  }
						}

						/* Small screens (phones, min-width: 768px) */
						@media (min-width: 768px) and (max-width: 991.98px) {
						  .col-md-6 {
							        width: 49%;
									float: right;
									box-sizing: border-box;
									margin: 0 0.5%;
						  }
						}

                     </style>
                     <script>
                        const totalElement = document.querySelector('.order_details .total');
                        // Remove the element if it exists
                        if (totalElement) {
                        totalElement.remove();
                        }

                       
                        const achFeeTotal =" . json_encode(number_format((float)$ach_service_fee + (float)$ach_convenience_service_fee, 2, '.', ''))  . ";
                        const creditFeeTotal =" . json_encode(number_format((float)$credit_service_fee + (float)$credit_convenience_service_fee, 2, '.', ''))  . ";
                        const applyCreditService =" . json_encode($options['apply_credit_service'] == 'yes' ? true : false) . ";
                        const applyCreditConv =" . json_encode($options['apply_credit_convenience_service'] == 'yes' ? true : false) . ";
                        const applyAchService =" . json_encode($options['apply_ach_service'] == 'yes' ? true : false) . ";
                        const applyAchConv =" . json_encode($options['apply_ach_convenience_service'] == 'yes' ? true : false) . ";
                        const mode = " . json_encode($this->mode == 'true' ? true : false) . ";
                        const acceptCredit = " . json_encode($this->accept_credit == 'yes' ? true : false) . ";
                        const acceptACH = " . json_encode($this->accept_ach == 'yes' ? true : false) . ";
                        const achServiceFee = " . json_encode(number_format((float)$ach_service_fee, 2, '.', ''))  . ";
                        const creditServiceFee = " . json_encode(number_format((float)$credit_service_fee, 2, '.', ''))  . ";
                        const achConFee = " . json_encode(number_format((float)$ach_convenience_service_fee, 2, '.', '')) . ";
                        const creditConFee = " . json_encode(number_format((float)$credit_convenience_service_fee, 2, '.', '')) . ";
                        const achFeeLabel = " . json_encode($ach_fee_label) . ";
                        const creditFeeLabel = " . json_encode($credit_fee_label) . ";
                        const creditTotal = " . json_encode(number_format((float)$credit_total, 2, '.', '')) . ";
                        const achTotal = " . json_encode(number_format((float)$ach_total, 2, '.', '')) . ";
                        const creditConFeeLabel =" . json_encode($credit_con_fee_label) . ";
                        const achConFeeLabel =" . json_encode($ach_con_fee_label) . ";
                        const currencySymbol =" . json_encode($currency_symbol) . ";
                        const Tax =" . json_encode($order_tax) . ";
                     </script>";


      $subtotal_row = '<tr id="sub-total-row"><td>Cart Total</td><td>' . $currency_symbol . (number_format((float)$order_total, 2, '.', ''))  . '</td></tr>';
      $tax_row = '<tr id="tax-row"><td>Tax</td><td>' . $currency_symbol . (number_format((float)$order_tax + $order_ship_tax, 2, '.', ''))  . '</td></tr>';
      $shipping_row = '<tr id="shipping-row"><td>Shipping</td><td>' . $currency_symbol .(number_format((float)$order_ship, 2, '.', '')) . '</td></tr>';
      $total_row = '<tr id="total-row" style="font-weight:bold"><td>Subtotal</td><td>' . $currency_symbol . (number_format((float)$sub_total, 2, '.', '')) . '</td></tr>';
      $discount_row =  $order->get_total_discount() ? '<tr id="discount-row"><td>Discount</td><td>-' . $currency_symbol . number_format((float)$order->get_total_discount(), 2, '.', '') . '</td></tr>' : '';

      $service_fee_row = '<tr id="service-fee-row"><td>' . $credit_fee_label . '</td><td>' . $currency_symbol . $credit_service_fee . '</td></tr>';
      $conv_fee_row = '<tr id="conv-fee-row"><td>' . $credit_con_fee_label . '</td><td>' . $currency_symbol . $credit_convenience_service_fee . '</td></tr>';
      $final_amount_row = '<tr id="final-amount-row" style="font-weight:bold"><td>Final Amount</td><td>' . $currency_symbol . $credit_total . '</td></tr>';

      $html_form .= '<div class="col-md-6" ><p style="margin-bottom: 10px;"><label style="margin: 20px 0 0 0;font-weight: 500;
">Order Summary </label></p><table id="additional-amount"><tbody>'
         . $subtotal_row
         . $discount_row
         . $shipping_row
         . $tax_row
         . $total_row
         . $service_fee_row
         . $conv_fee_row
         . $final_amount_row
         . '</tbody></table></div>';


      $html_form    .= '<form class="col-md-6" action="' . $processURI . '" method="post" id="chargeanywhere_payment_form" >'
         . '<div id="payment-form-root"></div>'


         . implode('', $authorize_args_array)
         . '<input type="submit" class="button" id="submit_chargeanywhere_payment_form" value="' . __('Pay via Charge Anywhere', 'chargeanywhere-woocommerce') . '" /> <a class="button cancel" href="' . $order->get_cancel_order_url() . '">' . __('Cancel order &amp; restore cart', 'chargeanywhere-woocommerce') . '</a>'

         . '</form>'
         . "<script  src='" . plugins_url('../assets/js/bundle.js', __FILE__) . "'></script>";
      return $html_form;
   }

   public function seed_gen()
   {
      return rand(1, 9) . time();
   }

   /**
    * To log the request & response data
    */
   public function logData($type, $data)
   {
      $date = new DateTime('now', new DateTimeZone(wp_timezone_string()));
      $data = array_merge(array('current_time' => $date->format("Y-m-d H:i:s-u")), $data);
      $fname = date("Y-m-d");
      $fileName = dirname(__FILE__) . "/logs/" . $fname . ".log";
      $file = fopen($fileName, "a") or die("Unable to open file!");
      fwrite($file, $type . ": " . json_encode($data) . "\n");
      fclose($file);
   }
}