<?php
/*
   Plugin Name: Charge Anywhere Payment Gateway For WooCommerce
   Description: Extends WooCommerce to Process Payments with Charge Anywhere version.
   Version: 4.2
   Plugin URI: https://www.chargeanywhere.com/
   Author: Charge Anywhere
   Author URI: https://www.chargeanywhere.com/
   License: Under GPL2
   WC requires at least: 8.3
   WC tested up to: 8.4
   WooCommerce HPOS Compatible: true
*/


// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


/**
 * WC ChargeAnywhere Payment gateway plugin class.
 *
 * @class WC_ChargeAnywhere_Payments
 */
class WC_ChargeAnywhere_Payments {

	/**
	 * Plugin bootstrapping.
	 */
	public static function init() {

		// ChargeAnywhere Payments gateway class.
		add_action( 'plugins_loaded', array( __CLASS__, 'includes' ), 0 );

		// Make the ChargeAnywhere Payments gateway available to WC.
		add_filter( 'woocommerce_payment_gateways', array( __CLASS__, 'add_gateway' ) );

		// Registers WooCommerce Blocks integration.
		add_action( 'woocommerce_blocks_loaded', array( __CLASS__, 'woocommerce_gateway_ChargeAnywhere_woocommerce_block_support' ) );

      add_action('before_woocommerce_init', function() {
            if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
                \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
            }
        });

      add_filter( 'woocommerce_custom_order_tables_compatible', function( $compatible, $plugin ) {
         if ( strpos( $plugin, 'chargeanywhere' ) !== false ) {
            return true;
         }
         return $compatible;
      }, 10, 2 );
      

	}

	/**
	 * Add the ChargeAnywhere Payment gateway to the list of available gateways.
	 *
	 * @param array
	 */
	public static function add_gateway( $gateways ) {

		$options = get_option( 'woocommerce_chargeanywhere_settings', array() );

		if ( isset( $options['hide_for_non_admin_users'] ) ) {
			$hide_for_non_admin_users = $options['hide_for_non_admin_users'];
		} else {
			$hide_for_non_admin_users = 'no';
		}

		if ( ( 'yes' === $hide_for_non_admin_users && current_user_can( 'manage_options' ) ) || 'no' === $hide_for_non_admin_users ) {
			$gateways[] = 'WC_Gateway_ChargeAnywhere';
		}
		return $gateways;
	}

	/**
	 * Plugin includes.
	 */
	public static function includes() {

		// Make the WC_Gateway_ChargeAnywhere class available.
		if ( class_exists( 'WC_Payment_Gateway' ) ) {
			require_once 'includes/class-wc-gateway-chargeanywhere.php';
		}
	}

	/**
	 * Plugin url.
	 *
	 * @return string
	 */
	public static function plugin_url() {
		return untrailingslashit( plugins_url( '/', __FILE__ ) );
	}

	/**
	 * Plugin url.
	 *
	 * @return string
	 */
	public static function plugin_abspath() {
		return trailingslashit( plugin_dir_path( __FILE__ ) );
	}

	/**
	 * Registers WooCommerce Blocks integration.
	 *
	 */
	public static function woocommerce_gateway_ChargeAnywhere_woocommerce_block_support() {
		if ( class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
			require_once 'includes/class-wc-chargeanywhere-payments-blocks.php';
			add_action(
				'woocommerce_blocks_payment_method_type_registration',
				function( Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry ) {
					$payment_method_registry->register( new WC_Gateway_ChargeAnywhere_Blocks_Support() );
				}
			);
		}
	}
}

WC_ChargeAnywhere_Payments::init();

   /**
    * Localisation
    */

   load_plugin_textdomain('wc-chargeanywhere', false, dirname(plugin_basename(__FILE__)) . '/languages');

   function custom_script_in_admin()
   {
      wp_register_script('CA_handler_script', plugin_dir_url(__FILE__) . 'includes/js/chargeanywhere-admin.js', '', true);
      wp_enqueue_script('CA_handler_script');
      wp_register_script('wc-chargeanywhere-script', plugin_dir_url(__FILE__) . '/includes/js/chargeanywhere-frontend-legacy.js', array('jquery', 'jquery-payment'), '4.0', true);
       wp_enqueue_script('wc-chargeanywhere-script');
   }

   add_action('admin_enqueue_scripts', 'custom_script_in_admin');
 
   /**
    * Fee Calculation and showing in frontend
    */
   add_action('woocommerce_cart_calculate_fees', function () {
      global $woocommerce;
      if (is_admin() && !defined('DOING_AJAX')) {
         return;
      }
      $args = array();
      if (isset($_POST['post_data'])) {
         parse_str($_POST['post_data'], $args);
      } else if (isset($_POST)) {
         $args = $_POST;
      }

      $chosen_payment_method = WC()->session->get('chosen_payment_method');
      $options = get_option('woocommerce_chargeanywhere_settings');
      WC()->session->set('credit_service_fee', 0);
      WC()->session->set('credit_convenience_service_fee', 0);
      WC()->session->set('ach_service_fee', 0);
      WC()->session->set('ach_convenience_service_fee', 0);

      if ($chosen_payment_method == 'chargeanywhere') {
         if (isset($args['chargeanywhere_option']) && $args['chargeanywhere_option'] == 'credit') {
            if ($options['accept_credit'] === 'yes' && $options['apply_credit_service'] === 'yes' && $options['credit_service_fee_value'] > 0) {
               $percentage = $options['credit_service_fee_value'] / 100;  // Percentage (5%) in float
               $totalTax = 0;
               $taxes = WC()->cart->get_cart_contents_taxes();
               foreach ($taxes as $tax) $totalTax += $tax;

               //WC()->cart->get_shipping_taxes() fetch only shipping taxes
               //WC()->cart->get_taxes() fetch all taxes
               //+ WC()->cart->get_shipping_total() shipping total func
               $percentage_fee = WC()->cart->get_cart_contents_total() * $percentage;
               if ($options['apply_credit_tax_amount_service_fee'] == 'yes')
                  $percentage_fee = (WC()->cart->get_cart_contents_total() + $totalTax) * $percentage;

               $percentage_fee = wc_format_decimal($percentage_fee);

               $label = " Service Fee";
               if (trim($options['credit_service_fee']) != '')
                  $label = " " . $options['credit_service_fee'];
               WC()->cart->add_fee(__($label, 'chargeanywhere-woocommerce'), $percentage_fee);
               WC()->session->set('credit_service_fee', $percentage_fee);
            }
            if ($options['accept_credit'] === 'yes' && $options['apply_credit_convenience_service'] === 'yes' && $options['credit_convenience_service_fee_value'] > 0) {
               $label = "Convenience Fee";
               if (trim($options['credit_convenience_service_fee']) != '')
                  $label = $options['credit_convenience_service_fee'];
               WC()->cart->add_fee(__($label, 'chargeanywhere-woocommerce'), $options['credit_convenience_service_fee_value']);
               WC()->session->set('credit_convenience_service_fee', $options['credit_convenience_service_fee_value']);
            }
         }
         if (isset($args['chargeanywhere_option']) && $args['chargeanywhere_option'] == 'ach') {
            if ($options['accept_ach'] === 'yes' && $options['apply_ach_service'] === 'yes' && $options['ach_service_fee_value'] > 0) {
               $percentage = $options['ach_service_fee_value'] / 100;  // Percentage (5%) in float

               $totalTax = 0;
               $taxes = WC()->cart->get_cart_contents_taxes();
               foreach ($taxes as $tax) $totalTax += $tax;

               $percentage_fee = WC()->cart->get_cart_contents_total() * $percentage;
               if ($options['apply_ach_tax_amount_service_fee'] == 'yes')
                  $percentage_fee = (WC()->cart->get_cart_contents_total() + $totalTax) * $percentage;

               $percentage_fee = wc_format_decimal($percentage_fee);
               $label = " Service Fee";
               if (trim($options['ach_service_fee']) != '')
                  $label = " " . $options['ach_service_fee'];

               WC()->cart->add_fee(__($label, 'chargeanywhere-woocommerce'), $percentage_fee);
               WC()->session->set('ach_service_fee', $percentage_fee);
            }
            if ($options['accept_ach'] === 'yes' && $options['apply_ach_convenience_service'] === 'yes' && $options['ach_convenience_service_fee_value'] > 0) {
               $label = "Convenience Fee";
               if (trim($options['ach_convenience_service_fee']) != '')
                  $label = $options['ach_convenience_service_fee'];

               WC()->cart->add_fee(__($label, 'chargeanywhere-woocommerce'), $options['ach_convenience_service_fee_value']);
               WC()->session->set('ach_convenience_service_fee', $options['ach_convenience_service_fee_value']);
            }
         }
      }
   });

   add_action('admin_footer', function () {
      $currentPostType = get_post_type();
      $order_id = get_the_ID();
      $order = wc_get_order($order_id);
      
      // Add null check before accessing the order
      if ($order) {
         $tran_meta = $order->get_meta('_chargeanywhere_transaction');
         $options = get_option('woocommerce_chargeanywhere_settings');
         if ($currentPostType != 'shop_order') return;
         
         // Check if $tran_meta is an array and has the required key
         if (!is_array($tran_meta) || !isset($tran_meta['chargeanywhere_option'])) {
            return;
         }
         
         if ($tran_meta['chargeanywhere_option'] == 'credit') {
            if ($options['credit_refund_service_fee'] == 'no')
               return;
         }
         if ($tran_meta['chargeanywhere_option'] == 'ach') {
            if ($options['ach_refund_service_fee'] == 'no')
               return;
         }
         
         // Basic refund settings for JavaScript
         $refund_settings = array(
            'credit_refund_service_fee' => $options['credit_refund_service_fee'] ?? 'yes',
            'ach_refund_service_fee' => $options['ach_refund_service_fee'] ?? 'yes'
         );
         
         // Get all Charge Anywhere managed fee items with their metadata
         $caw_managed_fees = array();
         
         foreach ($order->get_items('fee') as $item_id => $fee_item) {
            $fee_type = $fee_item->get_meta('_fee_type', true);
            $fee_name = $fee_item->get_name();
            
            if (in_array($fee_type, ['credit_service_fee', 'credit_convenience_fee', 'ach_service_fee', 'ach_convenience_fee'])) {
               // New orders with metadata - most reliable
               $caw_managed_fees[] = array(
                  'item_id' => $item_id,
                  'fee_type' => $fee_type,
                  'name' => $fee_name,
                  'method' => 'metadata'
               );
            } else {
               // Legacy orders without metadata - check if name matches current settings
               $is_legacy_fee = false;
               $legacy_fee_type = '';
               
               if ($tran_meta['chargeanywhere_option'] == 'credit') {
                  if (trim(mb_strtolower($fee_name)) == trim(mb_strtolower($options['credit_service_fee']))) {
                     $is_legacy_fee = true;
                     $legacy_fee_type = 'credit_service_fee';
                  } elseif (trim(mb_strtolower($fee_name)) == trim(mb_strtolower($options['credit_convenience_service_fee']))) {
                     $is_legacy_fee = true;
                     $legacy_fee_type = 'credit_convenience_fee';
                  }
               } elseif ($tran_meta['chargeanywhere_option'] == 'ach') {
                  if (trim(mb_strtolower($fee_name)) == trim(mb_strtolower($options['ach_service_fee']))) {
                     $is_legacy_fee = true;
                     $legacy_fee_type = 'ach_service_fee';
                  } elseif (trim(mb_strtolower($fee_name)) == trim(mb_strtolower($options['ach_convenience_service_fee']))) {
                     $is_legacy_fee = true;
                     $legacy_fee_type = 'ach_convenience_fee';
                  }
               }
               
               if ($is_legacy_fee) {
                  $caw_managed_fees[] = array(
                     'item_id' => $item_id,
                     'fee_type' => $legacy_fee_type,
                     'name' => $fee_name,
                     'method' => 'legacy_matching'
                  );
               }
            }
         }
      } else {
         return; // Exit if no valid order is found
      }
      ?>
      <script type="text/javascript">
         (function($) {
            // Robust fee identification using metadata - not brittle name patterns
            var cawManagedFees = <?php echo json_encode($caw_managed_fees); ?>;
            var refundSettings = <?php echo json_encode($refund_settings); ?>;
            
            // Enhanced fee refund protection against WordPress/WooCommerce interference
            function protectFeeRefunds() {
               // Disable fee refund inputs for Charge Anywhere managed fees
               $('#order_fee_line_items tr[data-order_item_id]').each(function() {
                  var $row = $(this);
                  var itemId = $row.attr('data-order_item_id');
                  
                  // Check if this item ID is in our managed fees list
                  var isManagedFee = false;
                  var feeInfo = null;
                  
                  for (var i = 0; i < cawManagedFees.length; i++) {
                     if (cawManagedFees[i].item_id == itemId) {
                        isManagedFee = true;
                        feeInfo = cawManagedFees[i];
                        break;
                     }
                  }
                  
                  if (isManagedFee) {
                     // Disable refund inputs for this managed fee
                     $row.find('.refund_line_total, .refund_line_tax').each(function() {
                        $(this).attr('disabled', 'disabled').addClass('caw-protected-fee');
                        
                        // Add visual indicator with fee type info
                        if (!$(this).next('.caw-fee-notice').length) {
                           var feeTypeLabel = feeInfo.fee_type.replace(/_/g, ' ').replace(/\b\w/g, function(l) { return l.toUpperCase(); });
                           var detectionMethod = feeInfo.method === 'metadata' ? 'Verified' : 'Legacy';
                           $(this).after('<small class="caw-fee-notice" style="color: #666; font-style: italic; display: block;">Managed by Charge Anywhere (' + feeTypeLabel + ' - ' + detectionMethod + ')</small>');
                        }
                     });
                     
                     // Also disable any edit inputs to prevent manual fee modification
                     $row.find('input[type="text"]').not('.caw-protected-fee').each(function() {
                        if ($(this).attr('name') && ($(this).attr('name').indexOf('order_item_name') > -1 || $(this).attr('name').indexOf('line_total') > -1)) {
                           $(this).attr('readonly', 'readonly').addClass('caw-protected-fee');
                           if (!$(this).next('.caw-fee-notice').length) {
                              $(this).after('<small class="caw-fee-notice" style="color: #666; font-style: italic; margin-left: 5px;">Protected</small>');
                           }
                        }
                     });
                  }
               });
            }
            
            // Run on page load
            $(document).ready(protectFeeRefunds);
            
            // Re-run when WooCommerce updates the order items (for dynamic updates)
            $(document).on('woocommerce_order_items_reload', protectFeeRefunds);
            
            // Re-run when refund UI is loaded/updated
            $(document).on('woocommerce_refund_items_loaded', protectFeeRefunds);
            
            // Enhanced refund UI improvements
            function enhanceRefundUI() {
               // Add refresh reminder for fully refunded items
               $('#order_line_items .item').each(function() {
                  var $row = $(this);
                  var $refundTotal = $row.find('.refund_line_total');
                  
                  if ($refundTotal.length && $refundTotal.val() && parseFloat($refundTotal.val()) > 0) {
                     // This item has been refunded - check if it's fully refunded
                     var $lineTotal = $row.find('.line_total');
                     var lineTotalVal = parseFloat($lineTotal.val() || $lineTotal.text().replace(/[^\d.-]/g, ''));
                     var refundTotalVal = parseFloat($refundTotal.val());
                     
                     if (Math.abs(lineTotalVal - Math.abs(refundTotalVal)) < 0.01) {
                        // Item is fully refunded
                        $row.addClass('fully-refunded-item');
                        if (!$row.find('.caw-refund-notice').length) {
                           $row.append('<small class="caw-refund-notice" style="color: #d63638; font-style: italic; display: block;">Fully Refunded - Refresh page if attempting new refund</small>');
                        }
                     }
                  }
               });
               
               // Add warning for potential double refund attempts
               $('#woocommerce-order-items').on('click', '.refund-items', function() {
                  if ($('.fully-refunded-item').length > 0) {
                     if (!confirm('Some items appear to be fully refunded. Are you sure you want to proceed? You may need to refresh the page first.')) {
                        return false;
                     }
                  }
               });
            }
            
            // Mutation observer for DOM changes (fallback protection)
            if (window.MutationObserver) {
               var observer = new MutationObserver(function(mutations) {
                  var shouldReprotect = false;
                  mutations.forEach(function(mutation) {
                     if (mutation.type === 'childList' && (mutation.target.id === 'order_fee_line_items' || mutation.target.id === 'order_line_items')) {
                        shouldReprotect = true;
                     }
                  });
                  if (shouldReprotect) {
                     setTimeout(function() {
                        protectFeeRefunds();
                        enhanceRefundUI();
                     }, 100); // Small delay for DOM stabilization
                  }
               });
               
               var targetNode = document.getElementById('order_fee_line_items');
               if (targetNode) {
                  observer.observe(targetNode, { childList: true, subtree: true });
               }
               
               var orderItemsNode = document.getElementById('order_line_items');
               if (orderItemsNode) {
                  observer.observe(orderItemsNode, { childList: true, subtree: true });
               }
            }
            
            // Run enhanced UI on load
            $(document).ready(function() {
               setTimeout(enhanceRefundUI, 500); // Delay to ensure WooCommerce has loaded
            });

            
         })(jQuery);
      </script>
      <?php
   });

   // Pass refund settings to JavaScript for fee calculations
   add_action('admin_footer', function () {
      if (!is_admin()) {
         return;
      }
      
      $gateway = new WC_Gateway_ChargeAnywhere();
      $settings = $gateway->settings;
      
      // Get current order's payment method
      $order_id = 0;
      $payment_method = '';
      
      if (isset($_GET['post'])) {
         $order_id = absint($_GET['post']);
      } elseif (isset($_GET['id'])) {
         $order_id = absint($_GET['id']);
      }
      
      $fee_types = array();
      
      if ($order_id) {
         $order = wc_get_order($order_id);
         if ($order && $order->get_payment_method() === 'chargeanywhere') {
            $tran_meta = $order->get_meta('_chargeanywhere_transaction');
            $payment_method = isset($tran_meta['chargeanywhere_option']) ? $tran_meta['chargeanywhere_option'] : '';
            
            // Get fee types from order metadata
            foreach ($order->get_fees() as $item_id => $fee_item) {
               $fee_type = $fee_item->get_meta('_fee_type');
               if ($fee_type) {
                  $fee_types[$item_id] = $fee_type;
               } else {
                  // Fallback: try to determine fee type from name
                  $fee_name = $fee_item->get_name();
                  if (strpos($fee_name, 'SF_') !== false || stripos($fee_name, 'service') !== false) {
                     $fee_types[$item_id] = 'service';
                  } elseif (strpos($fee_name, 'CF_') !== false || stripos($fee_name, 'convenience') !== false) {
                     $fee_types[$item_id] = 'convenience';
                  }
               }
            }
         }
      }
      
       ?>
       <script type="text/javascript">
       window.cawRefundSettings = {
          paymentMethod: '<?php echo esc_js($payment_method); ?>',
          creditRefundConvenienceFee: <?php echo ($settings['credit_refund_convenience_fee'] === 'yes') ? 'true' : 'false'; ?>,
          achRefundConvenienceFee: <?php echo ($settings['ach_refund_convenience_fee'] === 'yes') ? 'true' : 'false'; ?>,
          creditRefundServiceFee: <?php echo ($settings['credit_refund_service_fee'] === 'yes') ? 'true' : 'false'; ?>,
          achRefundServiceFee: <?php echo ($settings['ach_refund_service_fee'] === 'yes') ? 'true' : 'false'; ?>,
          feeTypes: <?php echo json_encode($fee_types); ?>
       };
       </script>
   <?php
   });

   add_action('woocommerce_review_order_before_payment', function () {
      ?><script type="text/javascript">
      (function($) {
      $('form.checkout').on('change', 'input[name^="payment_method"]', function() {
         $('body').trigger('update_checkout');
      });
      })(jQuery);
      </script><?php
   });

 
         add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'chargeAnyWhereSIP_action_links');
         function chargeAnyWhereSIP_action_links($links)
         {
            $chargeSIP_links = array(
               '<a href="admin.php?page=wc-settings&tab=checkout&section=chargeanywhere" >Settings</a>',
               '<a href="http://www.chargeanywhere.com" target="_blank">Support</a>'
            );
            return array_merge($links, $chargeSIP_links);
         }

         // remove fees sort
         add_action('woocommerce_sort_fees_callback', 'ca_sort_fees', 10, 3);
         function ca_sort_fees($order, $a, $b)
         {
            return $a->name > $b->name ? 1 : -1;
         }
         
         // Smart validation bypass for legitimate edge cases
         add_filter('caw_bypass_double_refund_validation', function($bypass, $order, $error) {
            // Only bypass in specific edge cases - the main validation should work now
            // This filter can be used for specific debugging or edge cases if needed
            return $bypass; // Default: don't bypass
         }, 10, 3);